package ru.yandex.travel.orders.services.orders;

import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.Hibernate;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderAggregateState;
import ru.yandex.travel.orders.entities.TrainOrderStateImErrorInfo;
import ru.yandex.travel.orders.proto.TOrderAggregateState;
import ru.yandex.travel.orders.proto.TTrainOrderAggregateStateExtInfo;
import ru.yandex.travel.orders.repository.OrderAggregateStateChangeRepository;
import ru.yandex.travel.orders.repository.OrderAggregateStateRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Component
@Slf4j
@RequiredArgsConstructor
public class OrderAggregateStateRefresher {

    private static final Set<EOrderType> ACCEPTED_ORDER_TYPES = Set.of(
            EOrderType.OT_TRAIN, EOrderType.OT_HOTEL_EXPEDIA, EOrderType.OT_GENERIC,
            EOrderType.OT_AVIA_AEROFLOT, EOrderType.OT_BUS // this two just for display state
    );

    private final OrderAggregateStateMapper orderAggregateStateMapper;

    private final OrderAggregateStateRepository orderAggregateStateRepository;

    private final OrderRepository orderRepository;

    private final OrderAggregateStateChangeRepository orderAggregateStateChangeRepository;

    private final Counter successCounter = Counter.builder("aggregate_state_refresh").tag("result", "success")
            .register(Metrics.globalRegistry);

    private final Counter errorCounter = Counter.builder("aggregate_state_refresh").tag("result", "error")
            .register(Metrics.globalRegistry);


    @TransactionMandatory
    public void refreshOrderAggregateState(UUID orderId) {
        Long maxChangeId = null;
        EOrderType orderType = null;
        try {
            log.debug("Refreshing aggregate state for order: {}, max change id", orderId);
            maxChangeId = orderAggregateStateChangeRepository.findMaxIdForOrder(orderId);
            log.debug("Max change id: {}", maxChangeId);
            OrderAggregateState orderAggregateState = null;
            Order order = orderRepository.getOne(orderId);

            orderType = order.getPublicType();
            if (!ACCEPTED_ORDER_TYPES.contains(orderType)) {
                log.debug("Skipping unsupported order type to refresh: {}", order.getPublicType());
                return;
            }

            order = (Order) Hibernate.unproxy(order);

            Optional<OrderAggregateState> maybeOrderAggregateState = orderAggregateStateRepository.findById(orderId);
            Order finalOrder = order;
            orderAggregateState = maybeOrderAggregateState.orElseGet(() -> {
                OrderAggregateState r = new OrderAggregateState();
                r.setId(orderId);
                r.setOrderPrettyId(finalOrder.getPrettyId());
                r.setOrderType(finalOrder.getPublicType());
                return orderAggregateStateRepository.save(r);
            });
            TOrderAggregateState protoOrderAggregateState = orderAggregateStateMapper.mapAggregateStateFromOrder(order);
            log.debug("Aggregate state for order id: {}, {}", orderId, protoOrderAggregateState);

            // common fields
            orderAggregateState.setOrderDisplayState(protoOrderAggregateState.getDisplayOrderState());
            orderAggregateState.setPaymentUrl(protoOrderAggregateState.getPaymentUrl());
            orderAggregateState.setPaymentErrorDisplayState(protoOrderAggregateState.getPaymentError());

            if (protoOrderAggregateState.hasReservedTo()) {
                orderAggregateState.setReservedTo(ProtoUtils.toInstant(protoOrderAggregateState.getReservedTo()));
            } else {
                orderAggregateState.setReservedTo(null);
            }
            // generic fields
            orderAggregateState.setGenericOrderAggregateState(protoOrderAggregateState.getGenericOrderAggregateState());
            orderAggregateState.setVersionHash(protoOrderAggregateState.getVersionHash());

            // hotel fields
            orderAggregateState.setHotelOrderAggregateState(protoOrderAggregateState.getHotelOrderAggregateState());

            // train fields

            orderAggregateState.setTrainOrderAggregateState(protoOrderAggregateState.getTrainOrderAggregateState());

            if (protoOrderAggregateState.hasTrainAggregateExtInfo()) {
                TTrainOrderAggregateStateExtInfo extInfo = protoOrderAggregateState.getTrainAggregateExtInfo();
                orderAggregateState.setTrainOrderInsuranceAggregateState(extInfo.getInsuranceState());

                if (extInfo.hasErrorInfo()) {
                    orderAggregateState.setHasTrainError(true);
                    orderAggregateState.setTrainErrorMessage(extInfo.getErrorInfo().getMessage());
                    orderAggregateState.setTrainErrorMessageCode(extInfo.getErrorInfo().getMessageCode());
                    orderAggregateState.setTrainImErrorInfo(new TrainOrderStateImErrorInfo());
                    orderAggregateState.getTrainImErrorInfo().setMessageParams(extInfo.getErrorInfo().getMessageParamList());
                    orderAggregateState.setTrainErrorType(extInfo.getErrorInfo().getErrorType());
                } else {
                    orderAggregateState.setHasTrainError(false);
                }

                if (extInfo.hasMaxPendingTill()) {
                    orderAggregateState.setMaxPendingTill(ProtoUtils.toInstant(extInfo.getMaxPendingTill()));
                } else {
                    orderAggregateState.setMaxPendingTill(null);
                }
            }
            successCounter.increment();
        } catch (Exception e) {
            log.error("Error refreshing order aggregate state for order with id: {}", orderId, e);
            errorCounter.increment();
        } finally {
            if (maxChangeId == null) {
                // there are no existing changes, the method is called for the first time, nothing to delete yet
                //noinspection ReturnInsideFinallyBlock
                return;
            }
            try {
                log.debug("Removing all order state changes for order {} before {}", orderId, maxChangeId);
                orderAggregateStateChangeRepository.cleanupByOrderIdAndIdLessThanEqual(orderId, maxChangeId);
            } catch (Exception ex) {
                log.error("Error deleting order aggregate state changes for id: {}", orderId, ex);
            }
        }
    }
}
