package ru.yandex.travel.orders.services.orders;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;

import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.suburban.partners.SuburbanProvider;

public class OrderCompatibilityUtils {
    public static final Set<EServiceType> HOTEL_SERVICE_TYPES = Set.of(
            EServiceType.PT_BNOVO_HOTEL,
            EServiceType.PT_DOLPHIN_HOTEL,
            EServiceType.PT_EXPEDIA_HOTEL,
            EServiceType.PT_TRAVELLINE_HOTEL,
            EServiceType.PT_BRONEVIK_HOTEL
    );

    public static boolean isConfirmed(Order order) {
        if (order instanceof GenericOrder) {
            return ((GenericOrder) order).getState() == EOrderState.OS_CONFIRMED;
        } else if (order instanceof HotelOrder) {
            return ((HotelOrder) order).getState() == EHotelOrderState.OS_CONFIRMED;
        } else if (order instanceof TrainOrder) {
            return ((TrainOrder) order).getState() == ETrainOrderState.OS_CONFIRMED;
        } else if (order instanceof AeroflotOrder) {
            return ((AeroflotOrder) order).getState() == EAeroflotOrderState.OS_CONFIRMED;
        } else {
            throw new RuntimeException("Unsupported order: type " + order.getPublicType() + ", class "
                    + order.getClass().getSimpleName());
        }
    }

    public static OrderItem getOnlyOrderItem(Order order) {
        return getOnlyOrderItem(order.getOrderItems());
    }

    public static <T extends OrderItem> T getOnlyOrderItem(List<T> orderItems) {
        Preconditions.checkArgument(orderItems.size() == 1,
                "Exactly 1 order item is expected but got %s", orderItems.size());
        return orderItems.get(0);
    }

    public static OrderItem getOnlyOptionalOrderItem(List<OrderItem> orderItems) {
        Preconditions.checkArgument(orderItems.size() <= 1,
                "Exactly 1 order item is expected but got %s", orderItems.size());
        return orderItems.isEmpty() ? null : orderItems.get(0);
    }

    public static void ensureHotelOrder(Order order) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isHotelOrder(order), "" +
                "Not a hotel order; type " + order.getPublicType() + ", class " + order.getClass().getSimpleName());
    }

    public static boolean isHotelOrder(Order order) {
        List<OrderItem> orderItems = order.getOrderItems();
        // using service types as tests don't use HotelOrderItem instances in some cases
        return orderItems.size() == 1 && HOTEL_SERVICE_TYPES.contains(orderItems.get(0).getPublicType());
    }

    public static HotelOrderItem getOnlyHotelOrderItem(Order order) {
        OrderItem orderItem = getOnlyOrderItem(order);
        Preconditions.checkArgument(orderItem instanceof HotelOrderItem,
                "HotelOrderItem is expected but got %s", orderItem.getClass().getName());
        return (HotelOrderItem) orderItem;
    }

    public static boolean isTrainOrder(Order order) {
        List<OrderItem> orderItems = order.getOrderItems();
        return !orderItems.isEmpty() && orderItems.stream()
                .allMatch(oi -> oi.getPublicType() == EServiceType.PT_TRAIN);
    }

    public static List<TrainOrderItem> getTrainOrderItems(Order order) {
        Preconditions.checkArgument(isTrainOrder(order),
                "Train order is expected but got %s", order.getClass().getSimpleName());
        return order.getOrderItems().stream()
                .filter(oi -> oi instanceof TrainOrderItem)
                .map(oi -> (TrainOrderItem) oi)
                .collect(Collectors.toList());
    }

    public static List<BusOrderItem> getBusOrderItems(Order order) {
        Preconditions.checkArgument(isBusOrder(order),
                "Bus order is expected but got %s", order.getClass().getSimpleName());
        return order.getOrderItems().stream()
                .filter(oi -> oi instanceof BusOrderItem)
                .map(oi -> (BusOrderItem) oi)
                .collect(Collectors.toList());
    }

    // todo(tlg-13,ganintsev): we need to get rid of this method before the complex train order is released to prod
    public static TrainOrderItem getOnlyTrainOrderItem(Order order) {
        return getOnlyOrderItem(getTrainOrderItems(order));
    }

    public static boolean isSuburbanOrder(Order order) {
        List<OrderItem> orderItems = order.getOrderItems();
        return !orderItems.isEmpty() && orderItems.stream().allMatch(oi -> oi.getPublicType() == EServiceType.PT_SUBURBAN);
    }

    public static boolean isAeroexpressOrder(Order order) {
        List<OrderItem> orderItems = order.getOrderItems();
        if (orderItems.size() != 1 || orderItems.get(0).getPublicType() != EServiceType.PT_SUBURBAN)
            return false;
        return ((SuburbanOrderItem)orderItems.get(0)).getPayload().getProviderCode() == SuburbanProvider.AEROEXPRESS;
    }

    public static boolean isBusOrder(Order order) {
        List<OrderItem> orderItems = order.getOrderItems();
        return !orderItems.isEmpty() && orderItems.stream().allMatch(oi -> oi.getPublicType() == EServiceType.PT_BUS);
    }

    public static SuburbanOrderItem getSuburbanOrderItem(Order order) {
        Preconditions.checkArgument(isSuburbanOrder(order),
                "Suburban order is expected but got %s", order.getClass().getSimpleName());
        var items = order.getOrderItems().stream()
                .filter(oi -> oi instanceof SuburbanOrderItem)
                .map(oi -> (SuburbanOrderItem) oi)
                .collect(Collectors.toList());

        Preconditions.checkArgument(items.size() == 1,
                "Suburban order must have 1 item, but has %s", items.size());

        return items.get(0);
    }
}
