package ru.yandex.travel.orders.services.orders;

import java.util.Set;
import java.util.UUID;

import lombok.extern.slf4j.Slf4j;
import org.hibernate.event.spi.PostUpdateEvent;
import org.hibernate.event.spi.PostUpdateEventListener;
import org.hibernate.persister.entity.EntityPersister;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderAggregateStateChange;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.PendingInvoice;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.workflow.entities.Workflow;

@Slf4j
public class OrderUpdateStateChangeEventListener implements PostUpdateEventListener {
    private static final Set<String> orderSet = Set.of(
            WellKnownWorkflowEntityType.HOTEL_ORDER.getDiscriminatorValue(),
            WellKnownWorkflowEntityType.TRAIN_ORDER.getDiscriminatorValue(),
            WellKnownWorkflowEntityType.GENERIC_ORDER.getDiscriminatorValue()
    );

    @Override
    public void onPostUpdate(PostUpdateEvent event) {
        try {
            final Object entity = event.getEntity();

            if (entity instanceof Order) {
                Order order = (Order) entity;
                insertIntoChanges(event, order.getId());
            } else if (entity instanceof OrderItem) {
                OrderItem orderItem = (OrderItem) entity;
                if (orderItem.getOrder() != null) {
                    insertIntoChanges(event, orderItem.getOrder().getId());
                }
            } else if (entity instanceof Invoice) {
                Invoice inv = (Invoice) entity;
                if (inv.getOrder() != null) {
                    insertIntoChanges(event, inv.getOrder().getId());
                }
            } else if (entity instanceof Workflow) {
                Workflow workflow = (Workflow) entity;
                if (orderSet.contains(workflow.getEntityType())) {
                    insertIntoChanges(event, workflow.getEntityId());
                }
            } else if (entity instanceof PaymentSchedule) {
                PaymentSchedule paymentSchedule = (PaymentSchedule) entity;
                insertIntoChanges(event, paymentSchedule.getOrder().getId());
            } else if (entity instanceof PendingInvoice) {
                PendingInvoice pendingInvoice = (PendingInvoice) entity;
                if (pendingInvoice.getOwnerOrder() != null) {
                    insertIntoChanges(event, pendingInvoice.getOwnerOrder().getId());
                }
            }
        } catch (Exception e) {
            log.error("An exception has occurred while post update", e);
        }
    }

    private void insertIntoChanges(PostUpdateEvent event, UUID orderId) {
        var change = OrderAggregateStateChange.createOrderAggregateStateChange(orderId);
        event.getSession().persist(change);
    }

    @Override
    public boolean requiresPostCommitHanding(EntityPersister persister) {
        return false;
    }

    @Override
    public boolean requiresPostCommitHandling(EntityPersister persister) {
        return false;
    }
}
