package ru.yandex.travel.orders.services.orders;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.protobuf.InvalidProtocolBufferException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.bus.service.BusPartKeyService;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.BusTicketRefund;
import ru.yandex.travel.orders.entities.FiscalReceipt;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.GenericOrderUserRefund;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.ERefundPartState;
import ru.yandex.travel.orders.proto.ERefundPartType;
import ru.yandex.travel.orders.proto.TDownloadBlankToken;
import ru.yandex.travel.orders.proto.TRefundInfo;
import ru.yandex.travel.orders.proto.TRefundPartContext;
import ru.yandex.travel.orders.proto.TRefundPartInfo;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.EBusTicketRefundState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;
import ru.yandex.travel.orders.workflow.train.proto.TTrainDownloadBlankParams;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.service.TrainPartKeyService;

@Service
@Slf4j
@RequiredArgsConstructor
public class RefundPartsService {
    private final Clock clock;

    private final static Set<EOrderState> REFUND_PARTS_ENABLED_ORDER_STATES = Set.of(
            EOrderState.OS_REFUNDED,
            EOrderState.OS_REFUNDING,
            EOrderState.OS_CONFIRMED);
    private final static Set<EOrderItemState> REFUND_PARTS_ENABLED_TRAIN_ITEM_STATES = Set.of(
            EOrderItemState.IS_REFUNDED,
            EOrderItemState.IS_CONFIRMED);
    private final static Set<ImBlankStatus> NON_REFUNDABLE_IM_BLANK_STATUSES = Set.of(
            ImBlankStatus.STRICT_BOARDING_PASS,
            ImBlankStatus.REFUNDED
    );
    private final static int DAYS_AFTER_DEPARTURE_TO_BECOME_ARCHIVED = 10;

    public static String partContextToString(TRefundPartContext ctx) {
        return Base64.getEncoder().encodeToString(ctx.toByteArray());
    }

    public List<TRefundPartInfo> getTrainServiceRefundParts(
            TrainOrderItem orderItem,
            TrainOrderItem backwardWayItem,
            Map<String, TRefundInfo> refunds,
            Map<UUID, Object> updatedOrderItemsPayloads
    ) {
        List<TRefundPartInfo> result = new ArrayList<>();
        if (!REFUND_PARTS_ENABLED_TRAIN_ITEM_STATES.contains(orderItem.getState())) {
            return result;
        }
        TrainReservation payload = getTrainReservation(orderItem, updatedOrderItemsPayloads);
        TrainReservation payloadBack = getTrainReservation(backwardWayItem, updatedOrderItemsPayloads);
        Map<Integer, TrainPassenger> backwardPassengers = new HashMap<>();
        if (!payload.backwardWayInRoundTripWithDiscount() && payloadBack != null) {
            backwardPassengers = payloadBack.getPassengers().stream().collect(Collectors
                    .toMap(TrainPassenger::getCustomerId, x -> x));
        }
        List<Integer> adultBlanks = payload.getPassengers().stream()
                .filter(p -> p.getCategory() != PassengerCategory.BABY)
                .map(p -> p.getTicket().getBlankId()).collect(Collectors.toList());
        for (TrainPassenger p : payload.getPassengers()) {
            var context = TRefundPartContext.newBuilder();
            var trainContext = context.getTrainTicketPartContextBuilder();
            context.setServiceId(orderItem.getId().toString());
            context.setType(ERefundPartType.RPT_SERVICE_PART);
            trainContext.setCustomerId(p.getCustomerId());
            trainContext.setBlankId(p.getTicket().getBlankId());
            var partCtxStr = partContextToString(context.build());
            var partKey = TrainPartKeyService.getPassengerKey(orderItem.getId(), p.getCustomerId());
            var part = TRefundPartInfo.newBuilder()
                    .setType(ERefundPartType.RPT_SERVICE_PART)
                    .setKey(partKey)
                    .setContext(partCtxStr);

            boolean haveTimeToReturn = p.getTicket() != null && Optional.ofNullable(p.getTicket().calculateCanReturnTill(payload))
                    .map(canReturnTill -> canReturnTill.isAfter(Instant.now(clock)))
                    .orElse(true);
            boolean haveTimeToReturnOffline = p.getTicket() != null && Optional.ofNullable(p.getTicket().calculateCanReturnTill(payload))
                    .map(canReturnTill -> canReturnTill.isAfter(Instant.now(clock).minus(Duration.ofDays(DAYS_AFTER_DEPARTURE_TO_BECOME_ARCHIVED))))
                    .orElse(true);
            var backwardPassenger = backwardPassengers.get(p.getCustomerId());
            if (backwardPassenger != null && backwardPassenger.getTicket().getRefundStatus() != TrainTicketRefundStatus.REFUNDED) {
                part.setState(ERefundPartState.RPS_DISABLED);
            } else if (p.getTicket() == null) {
                part.setState(ERefundPartState.RPS_DISABLED);
            } else if (p.getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED) {
                part.setState(ERefundPartState.RPS_REFUNDED);
            } else if (NON_REFUNDABLE_IM_BLANK_STATUSES.contains(p.getTicket().getImBlankStatus())) {
                part.setState(ERefundPartState.RPS_DISABLED);
            } else if (!haveTimeToReturn && !haveTimeToReturnOffline) {
                part.setState(ERefundPartState.RPS_DISABLED);
            } else if (!haveTimeToReturn && haveTimeToReturnOffline) {
                part.setState(payload.isProviderP2() ? ERefundPartState.RPS_DISABLED : ERefundPartState.RPS_OFFLINE_ENABLED);
            } else if (p.getCategory() == PassengerCategory.BABY && adultBlanks.contains(p.getTicket().getBlankId())) {
                part.setState(ERefundPartState.RPS_DEPENDENT);
            } else if (p.getTicket().getRefundStatus() == null &&
                    orderItem.getState() == EOrderItemState.IS_CONFIRMED) {
                part.setState(ERefundPartState.RPS_ENABLED);
            } else {
                part.setState(ERefundPartState.RPS_DISABLED);
            }
            if (refunds.containsKey(partKey)) {
                part.setRefund(refunds.get(partKey));
            }
            result.add(part.build());
        }
        return result;
    }

    private static TrainReservation getTrainReservation(TrainOrderItem orderItem,
                                                        Map<UUID, Object> updatedOrderItemsPayloads) {
        if (orderItem == null) {
            return null;
        }
        if (updatedOrderItemsPayloads != null && updatedOrderItemsPayloads.containsKey(orderItem.getId())) {
            return (TrainReservation) updatedOrderItemsPayloads.get(orderItem.getId());
        }
        return orderItem.getPayload();
    }

    public static TRefundPartContext partContextFromString(String ctx) {
        try {
            return TRefundPartContext.parseFrom(Base64.getDecoder().decode(ctx));
        } catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Unable to deserialize refund part context", e);
        }
    }

    public List<TRefundPartInfo> getRefundParts(GenericOrder order, Map<UUID, Object> updatedOrderItemsPayloads) {
        List<TRefundPartInfo> result = new ArrayList<>();
        if (!REFUND_PARTS_ENABLED_ORDER_STATES.contains(order.getState())) {
            return result;
        }
        Map<String, TRefundInfo> refundsByKey = null;
        try {
            refundsByKey = getRefundsByKey(order);
        } catch (Exception e) {
            log.warn("Failed to retrieve order refunds for {}", order.getId(), e);
        }
        TrainOrderItem backwardWayTrainOrderItem = getBackwardWayTrainOrderItem(order);
        for (OrderItem orderItem : order.getOrderItems()) {
            var itemContext = TRefundPartContext.newBuilder()
                    .setServiceId(orderItem.getId().toString())
                    .setType(ERefundPartType.RPT_SERVICE);
            var itemPart = TRefundPartInfo.newBuilder()
                    .setContext(partContextToString(itemContext.build()))
                    .setType(ERefundPartType.RPT_SERVICE)
                    .setKey(TrainPartKeyService.getServiceKey(orderItem.getId()));
            switch (orderItem.getPublicType()) {
                case PT_TRAIN:
                    TrainOrderItem trainOrderItem = ((TrainOrderItem) orderItem);
                    List<TRefundPartInfo> serviceRefundParts = getTrainServiceRefundParts(trainOrderItem,
                            backwardWayTrainOrderItem, refundsByKey, updatedOrderItemsPayloads);
                    if (trainOrderItem.getPayload().isFullRefundPossible() &&
                            serviceRefundParts.stream().allMatch(x -> (x.getState() != ERefundPartState.RPS_DISABLED &&
                                    x.getState() != ERefundPartState.RPS_OFFLINE_ENABLED)) &&
                            serviceRefundParts.stream().anyMatch(x -> x.getState() == ERefundPartState.RPS_ENABLED)) {
                        if (trainOrderItem.getPayload().isOnlyFullReturnPossible()) {
                            serviceRefundParts = serviceRefundParts.stream()
                                    .filter(x -> x.getState() == ERefundPartState.RPS_ENABLED)
                                    .map(x -> x.toBuilder().setState(ERefundPartState.RPS_DISABLED).build())
                                    .collect(Collectors.toList());
                        }
                        itemPart.setState(ERefundPartState.RPS_ENABLED);
                    } else if (trainOrderItem.getPayload().isFullRefundPossible() &&
                            serviceRefundParts.stream().allMatch(x -> (x.getState() != ERefundPartState.RPS_DISABLED)) &&
                            serviceRefundParts.stream().anyMatch(x -> x.getState() == ERefundPartState.RPS_OFFLINE_ENABLED)) {
                        itemPart.setState(ERefundPartState.RPS_OFFLINE_ENABLED);
                    } else if (trainOrderItem.getState() == EOrderItemState.IS_REFUNDED) {
                        itemPart.setState(ERefundPartState.RPS_REFUNDED);
                    }  else {
                        itemPart.setState(ERefundPartState.RPS_DISABLED);
                    }
                    result.addAll(serviceRefundParts);
                    break;
                case PT_BUS:
                    BusOrderItem busOrderItem = (BusOrderItem) orderItem;
                    List<TRefundPartInfo> busServiceRefundParts = getBusServiceRefundParts(busOrderItem, refundsByKey);
                    result.addAll(busServiceRefundParts);
                    if (busServiceRefundParts.stream().allMatch(x -> x.getState() != ERefundPartState.RPS_DISABLED) &&
                            busServiceRefundParts.stream().anyMatch(x -> x.getState() == ERefundPartState.RPS_ENABLED)) {
                        itemPart.setState(ERefundPartState.RPS_ENABLED);
                    } else if (busOrderItem.getState() == EOrderItemState.IS_REFUNDED) {
                        itemPart.setState(ERefundPartState.RPS_REFUNDED);
                    } else {
                        itemPart.setState(ERefundPartState.RPS_DISABLED);
                    }
                    break;
                // TODO(tlg-13): case PT_HOTEL:
                default:
                    itemPart.setState(ERefundPartState.RPS_DISABLED);
            }
            result.add(itemPart.build());
        }
        var orderContext = TRefundPartContext.newBuilder().setType(ERefundPartType.RPT_ORDER);
        var orderPart = TRefundPartInfo.newBuilder()
                .setType(ERefundPartType.RPT_ORDER)
                .setContext(partContextToString(orderContext.build()));
        List<TRefundPartInfo> serviceRefunds = result.stream()
                .filter(x -> x.getType() == ERefundPartType.RPT_SERVICE)
                .collect(Collectors.toList());
        boolean orderRefundPossible =
                serviceRefunds.stream().allMatch(x -> x.getState() != ERefundPartState.RPS_DISABLED &&
                        x.getState() != ERefundPartState.RPS_OFFLINE_ENABLED) &&
                        serviceRefunds.stream().anyMatch(x -> x.getState() == ERefundPartState.RPS_ENABLED);

        boolean orderOfflineRefundPossible =
                serviceRefunds.stream().allMatch(x -> x.getState() != ERefundPartState.RPS_DISABLED) &&
                        serviceRefunds.stream().anyMatch(x -> x.getState() == ERefundPartState.RPS_OFFLINE_ENABLED);
        if (orderRefundPossible) {
            orderPart.setState(ERefundPartState.RPS_ENABLED);
        } else if (order.getState() == EOrderState.OS_REFUNDED) {
            orderPart.setState(ERefundPartState.RPS_REFUNDED);
        } else if (orderOfflineRefundPossible) {
            orderPart.setState(ERefundPartState.RPS_OFFLINE_ENABLED);
        } else {
            orderPart.setState(ERefundPartState.RPS_DISABLED);
        }
        result.add(orderPart.build());
        return result;
    }

    private List<TRefundPartInfo> getBusServiceRefundParts(BusOrderItem orderItem, Map<String, TRefundInfo> refunds) {
        List<TRefundPartInfo> result = new ArrayList<>();
        for (var ticket : orderItem.getPayload().getOrder().getTickets()) {
            var context = TRefundPartContext.newBuilder();
            var busContext = context.getBusRefundPartContextBuilder();
            busContext.setTicketId(ticket.getId());
            context.setServiceId(orderItem.getId().toString());
            context.setType(ERefundPartType.RPT_SERVICE_PART);
            var partCtxStr = partContextToString(context.build());
            var partKey = BusPartKeyService.getTicketKey(orderItem.getId(), ticket.getId());
            var part = TRefundPartInfo.newBuilder()
                    .setType(ERefundPartType.RPT_SERVICE_PART)
                    .setKey(partKey)
                    .setContext(partCtxStr);
            if (ticket.getRefundedPrice() != null) {
                part.setState(ERefundPartState.RPS_REFUNDED);
            } else if (orderItem.getState() == EOrderItemState.IS_CONFIRMED) {
                part.setState(ERefundPartState.RPS_ENABLED);
            } else {
                part.setState(ERefundPartState.RPS_DISABLED);
            }
            if (refunds.containsKey(partKey)) {
                part.setRefund(refunds.get(partKey));
            }
            result.add(part.build());
        }
        return result;
    }

    private TrainOrderItem getBackwardWayTrainOrderItem(GenericOrder order) {
        return order.getOrderItems().stream()
                .filter(x -> x.getPublicType() == EServiceType.PT_TRAIN)
                .map(x -> (TrainOrderItem) x)
                .filter(x -> x.getPayload().backwardWayInRoundTripWithDiscount())
                .findFirst().orElse(null);
    }

    private Map<String, TRefundInfo> getRefundsByKey(GenericOrder order) {
        Map<String, TRefundInfo> result = new HashMap<>();
        Map<UUID, List<String>> fiscalReceiptsByRefundId = order.getCurrentInvoice().getFiscalReceipts().stream()
                .filter(x -> x.getOrderRefundId() != null && x.getReceiptUrl() != null)
                .collect(Collectors.groupingBy(FiscalReceipt::getOrderRefundId,
                        Collectors.mapping(FiscalReceipt::getReceiptUrl, Collectors.toList())));
        UUID lastRefundId = order.getOrderRefunds().stream()
                .max(Comparator.comparing(x -> x.getCreatedAt().getEpochSecond()))
                .map(OrderRefund::getId).orElse(null);
        for (OrderRefund refund : order.getOrderRefunds()) {
            for (TrainTicketRefund ticketRefund : OrderUtils.getTrainTicketRefunds(refund)) {
                for (PassengerRefundInfo refundPassenger : ticketRefund.getPayload().getItems()) {
                    if ((ticketRefund.getState() == ETrainTicketRefundState.RS_FAILED ||
                            refundPassenger.getRefundOperationStatus() != ImOperationStatus.OK) &&
                            !refund.getId().equals(lastRefundId)) {
                        // TRAVELBACK-1937 договорились писать проваленные возвраты в IRefundPartInfo.refund,
                        // только если это был последний возврат
                        continue;
                    }
                    String key = TrainPartKeyService.getPassengerKey(ticketRefund.getOrderItem().getId(),
                            refundPassenger.getCustomerId());
                    var refundInfo = TRefundInfo.newBuilder()
                            .setId(refund.getId().toString())
                            .setRefundType(refund.getRefundType());
                    if (refund.getState() != EOrderRefundState.RS_REFUNDED && refund.getState() != EOrderRefundState.RS_FAILED) {
                        refundInfo.setState(refund.getState());
                    } else if (refundPassenger.getRefundOperationStatus() == ImOperationStatus.OK) {
                        refundInfo.setState(EOrderRefundState.RS_REFUNDED);
                        refundInfo.setRefundAmount(ProtoUtils.toTPrice(refundPassenger.calculateActualRefundSum()));
                        if (!refundPassenger.isDependent() && refundPassenger.getRefundOperationId() != null) {
                            refundInfo.setRefundBlankToken(getRefundBlankToken(
                                            order.getId(),
                                            ticketRefund.getOrderItem().getId(),
                                            refundPassenger.getRefundOperationId()
                                    )
                            );
                            if (fiscalReceiptsByRefundId.containsKey(refund.getId())) {
                                refundInfo.addAllPaymentRefundReceiptUrls(fiscalReceiptsByRefundId.get(refund.getId()));
                            }
                        }
                    } else {
                        refundInfo.setState(EOrderRefundState.RS_FAILED);
                    }
                    result.put(key, refundInfo.build());
                }
            }

            List<BusTicketRefund> busTicketRefunds = refund instanceof GenericOrderUserRefund
                    ? ((GenericOrderUserRefund) refund).getBusTicketRefunds() : List.of();
            for (BusTicketRefund ticketRefund : busTicketRefunds) {
                if (ticketRefund.getState() == EBusTicketRefundState.RS_FAILED &&
                        !refund.getId().equals(lastRefundId)) {
                    continue;
                }
                String key = BusPartKeyService.getTicketKey(ticketRefund.getOrderItem().getId(),
                        ticketRefund.getPayload().getTicketId());
                var refundInfo = TRefundInfo.newBuilder()
                        .setId(refund.getId().toString())
                        .setRefundType(refund.getRefundType());
                if (ticketRefund.getState() == EBusTicketRefundState.RS_REFUNDED) {
                    refundInfo.setState(EOrderRefundState.RS_REFUNDED);
                    refundInfo.setRefundAmount(ProtoUtils.toTPrice(ticketRefund.getPayload().getRefundAmount()));
                    if (fiscalReceiptsByRefundId.containsKey(refund.getId())) {
                        refundInfo.addAllPaymentRefundReceiptUrls(fiscalReceiptsByRefundId.get(refund.getId()));
                    }
                } else if (ticketRefund.getState() == EBusTicketRefundState.RS_FAILED) {
                    refundInfo.setState(EOrderRefundState.RS_FAILED);
                } else {
                    refundInfo.setState(refund.getState());
                }
                result.put(key, refundInfo.build());
            }
        }
        return result;
    }

    private TDownloadBlankToken getRefundBlankToken(UUID orderId, UUID serviceId, int refundOperationId) {
        return TDownloadBlankToken.newBuilder()
                .setOrderId(orderId.toString())
                .setTrainDownloadBlankParams(TTrainDownloadBlankParams.newBuilder()
                        .setServiceId(serviceId.toString())
                        .setOperationId(refundOperationId).build())
                .build();
    }
}
