package ru.yandex.travel.orders.services.orders.updates;

import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Component;

import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.async.AsyncProducerConfig;
import ru.yandex.kikimr.persqueue.producer.transport.message.inbound.ProducerInitResponse;
import ru.yandex.travel.commons.network.NetworkUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.workflow.order.proto.OrderUpdatedEvent;

@Component
@RequiredArgsConstructor
@Slf4j
public class OrdersUpdatesQueueProducerImpl implements OrdersUpdatesQueueProducer, AutoCloseable {
    private final LogbrokerClientFactory logbrokerClientFactory;
    private final OrdersUpdatesQueueProducerConfigurationProperties properties;

    @Nullable
    private volatile AsyncProducer asyncProducer;
    private volatile boolean closed;

    @SneakyThrows
    @Override
    public void enqueue(UUID orderId, Instant time) {
        try {
            var event = OrderUpdatedEvent.newBuilder()
                    .setOrderId(orderId.toString())
                    .setUpdatedAt(ProtoUtils.fromInstant(time))
                    .build();
            var producer = getOrCreateAsyncProducer();
            producer.write(event.toByteArray()).get();
        } catch (Exception e) {
            asyncProducer = null;
            throw e;
        }
    }

    public void init() {
        AsyncProducer asyncProducer = this.asyncProducer;
        if (asyncProducer == null) {
            asyncProducer = createAndInitAsyncProducer();
            this.asyncProducer = asyncProducer;
        }
    }

    private synchronized AsyncProducer getOrCreateAsyncProducer() {
        if (closed) {
            throw new IllegalStateException("LogbrokerEventProducer is closed");
        }

        AsyncProducer asyncProducer = this.asyncProducer;
        if (asyncProducer == null) {
            asyncProducer = createAndInitAsyncProducer();
            this.asyncProducer = asyncProducer;
        }
        return asyncProducer;
    }

    private AsyncProducer createAndInitAsyncProducer() {
        AsyncProducer asyncProducer = null;
        try {
            var sourceId = generateSourceId();
            var config = AsyncProducerConfig.builder(properties.getTopic(), sourceId)
                    .setCredentialsProvider(() -> Credentials.oauth(properties.getOauthToken()))
                    .build();
            asyncProducer = logbrokerClientFactory.asyncProducer(config);
            asyncProducer.closeFuture()
                    .whenComplete((result, exception) -> {
                        if (exception != null) {
                            log.error("async producer is closed", exception);
                        }
                        closeAsyncProducer();
                    });
            ProducerInitResponse init = asyncProducer.init().get(properties.getInitTimeout().toSeconds(),
                    TimeUnit.SECONDS);
            log.info("async producer initialized, sessionId={}, partition={}", init.getSessionId(),
                    init.getPartition());
            return asyncProducer;
        } catch (InterruptedException e) {
            if (asyncProducer != null) {
                asyncProducer.close();
            }
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        } catch (TimeoutException e) {
            asyncProducer.close();
            throw new RuntimeException(e);
        } catch (ExecutionException e) {
            throw new RuntimeException(e.getCause());
        }
    }

    private synchronized void closeAsyncProducer() {
        AsyncProducer asyncProducer = this.asyncProducer;
        this.asyncProducer = null;

        if (asyncProducer != null) {
            asyncProducer.close();
        }
    }

    @Override
    public void close() {
        closed = true;
        closeAsyncProducer();
    }

    private byte[] generateSourceId() {
        var randomFactor = 1000;
        return String.format("%s_%d",
                Math.abs(NetworkUtils.getLocalHostName().hashCode()),
                Math.round(Math.random() * randomFactor)).getBytes(StandardCharsets.UTF_8);
    }
}
