package ru.yandex.travel.orders.services.orders.updates;

import java.util.Collection;
import java.util.Set;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.orders.repository.OrdersUpdatesQueueStateChangeRepository;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorHelper;

@Configuration
@EnableConfigurationProperties(OrdersUpdatesQueueStateRefreshConfigurationProperties.class)
@RequiredArgsConstructor
public class OrdersUpdatesQueueStateRefreshConfiguration {
    private final OrdersUpdatesQueueStateChangeRepository ordersUpdatesQueueStateChangeRepository;

    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    private final OrdersUpdatesQueueStateRefreshConfigurationProperties ordersUpdatesQueueStateRefreshConfigurationProperties;

    @Bean
    @SneakyThrows
    public OrdersUpdatesQueueProducer ordersUpdatesQueueProducer(OrdersUpdatesQueueProducerFactory ordersUpdatesQueueProducerFactory) {
        var producer = ordersUpdatesQueueProducerFactory.create();
        producer.init();
        return producer;
    }

    @Bean
    public OrdersUpdatesQueueStateRefresher ordersUpdatesQueueStateRefresher(OrdersUpdatesQueueProducer ordersUpdatesQueueProducer) {
        return new OrdersUpdatesQueueStateRefresher(
                ordersUpdatesQueueProducer,
                ordersUpdatesQueueStateChangeRepository);
    }

    @Bean
    @ConditionalOnProperty("orders-updates-queue-state-refresh.task-processor.enabled")
    public TaskProcessor<UUID> ordersUpdatesQueueRefreshTaskProcessor(OrdersUpdatesQueueStateRefresher ordersUpdatesQueueStateRefresher) {
        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                var excludeIds = getExcludedIds();
                return ordersUpdatesQueueStateChangeRepository.getOrdersWaitingStateRefresh(excludeIds, maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                var excludedIds = getExcludedIds();
                return ordersUpdatesQueueStateChangeRepository.countOrdersWaitingStateRefresh(excludedIds);
            }

            private Set<UUID> getExcludedIds() {
                Set<UUID> excludedIds = getLockedTaskKeys();
                return excludedIds == null || excludedIds.isEmpty() ?
                        OrdersUpdatesQueueStateChangeRepository.NO_EXCLUDE_IDS : excludedIds;
            }
        };
        return new TaskProcessor<>(
                taskKeyProvider,
                ordersUpdatesQueueStateRefresher::refreshState,
                forcedRollbackTxManagerWrapper,
                TaskProcessorHelper.createDefaultTxDefinition("refreshOrdersUpdatesQueueState"),
                ordersUpdatesQueueStateRefreshConfigurationProperties.getTaskProcessor()
        );
    }
}
