package ru.yandex.travel.orders.services.orders.updates;

import java.util.UUID;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.OrdersUpdatesQueueStateChange;
import ru.yandex.travel.orders.repository.OrdersUpdatesQueueStateChangeRepository;
import ru.yandex.travel.tx.utils.TransactionMandatory;

/**
 * Поллит очередь событий "заказ создан/изменен":
 *  - отправляет последнее известное обновление X по заказу Y в logbroker
 *  - удаляет из очереди все обновления заказа Y с id <= id(X)
 */
@Slf4j
@RequiredArgsConstructor
public class OrdersUpdatesQueueStateRefresher {

    private final OrdersUpdatesQueueProducer ordersUpdatesQueueProducer;

    private final OrdersUpdatesQueueStateChangeRepository ordersUpdatesQueueStateChangeRepository;

    private final Counter successCounter = Counter.builder("orders_updates_queue_state_refresh").tag("result",
                    "success")
            .register(Metrics.globalRegistry);

    private final Counter errorCounter = Counter.builder("orders_updates_queue_state_refresh").tag("result", "error")
            .register(Metrics.globalRegistry);

    @TransactionMandatory
    public void refreshState(UUID orderId) {
        OrdersUpdatesQueueStateChange mostRecentChange = null;
        try (NestedMdc ignored = NestedMdc.forEntity(orderId, null)) {
            try {
                mostRecentChange = ordersUpdatesQueueStateChangeRepository.findMostRecentStateChangeForOrder(orderId);
                log.info("Refreshing orders updates queue state, change with id {} happend at: {}",
                        mostRecentChange.getId(), mostRecentChange.getCreatedAt());
                ordersUpdatesQueueProducer.enqueue(orderId, mostRecentChange.getCreatedAt());
                successCounter.increment();
            } catch (Exception e) {
                log.error("Error refreshing orders updates queue state", e);
                errorCounter.increment();
                mostRecentChange = null;
            } finally {
                if (mostRecentChange == null) {
                    // there are no existing changes, the method is called for the first time, nothing to delete yet
                    //noinspection ReturnInsideFinallyBlock
                    return;
                }
                try {
                    log.debug("Removing all orders updates queue state changes before {}", mostRecentChange.getId());
                    ordersUpdatesQueueStateChangeRepository.cleanupByOrderIdAndIdLessThanEqual(orderId,
                            mostRecentChange.getId());
                } catch (Exception ex) {
                    log.error("Error deleting orders updates queue state changes", ex);
                }
            }
        }
    }
}
