package ru.yandex.travel.orders.services.partners;

import java.time.Instant;
import java.util.NoSuchElementException;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.hotels.administrator.export.proto.ContractInfo;
import ru.yandex.travel.hotels.administrator.export.proto.HotelAgreement;
import ru.yandex.travel.hotels.common.partners.travelline.model.dto.HotelDto;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.cache.BalanceContractDictionary;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.EVat;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.entities.BronevikOrderItem;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.partners.AeroflotBillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.BillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.BillingPartnerConfig;
import ru.yandex.travel.orders.entities.partners.BronevikBillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.BusBillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreementProvider;
import ru.yandex.travel.orders.entities.partners.DolphinBillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.ExpediaBillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.SuburbanBillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.TrainBillingPartnerAgreement;
import ru.yandex.travel.orders.repository.BillingPartnerConfigRepository;
import ru.yandex.travel.orders.services.finances.HotelAgreementService;
import ru.yandex.travel.orders.services.finances.providers.DolphinFinancialDataProviderProperties;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnvProvider;
import ru.yandex.travel.orders.services.suburban.providers.SuburbanProviderBase;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.configuration.AeroflotWorkflowProperties;
import ru.yandex.travel.orders.workflows.orderitem.bronevik.BronevikProperties;
import ru.yandex.travel.orders.workflows.orderitem.bus.BusProperties;
import ru.yandex.travel.orders.workflows.orderitem.expedia.ExpediaProperties;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Service
@RequiredArgsConstructor
@Slf4j
public class BillingPartnerService {
    private final BillingPartnerConfigRepository partnerConfigRepository;
    private final ExpediaProperties expediaProperties;
    private final DolphinFinancialDataProviderProperties dolphinBillingProperties;
    private final BronevikProperties bronevikProperties;
    private final HotelAgreementService hotelAgreementService;
    private final Meters meters;
    private final AeroflotWorkflowProperties aeroflotWorkflowProperties;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final SuburbanOrderItemEnvProvider suburbanEnvProvider;
    private final BusProperties busProperties;
    private final BalanceContractDictionary balanceContractDictionary;

    @TransactionMandatory
    public boolean isPartnerAgreementActive(Long billingClientId) {
        BillingPartnerConfig config = partnerConfigRepository.findById(billingClientId).orElse(null);
        if (config == null) {
            throw new NoSuchElementException("No billing partner config for billing client id " + billingClientId);
        }
        return config.isAgreementActive();
    }

    @TransactionMandatory
    public void addAgreementOrThrow(OrderItem orderItem) {
        switch (orderItem.getPublicType()) {
            case PT_EXPEDIA_HOTEL:
                addExpediaAgreementOrThrow((ExpediaOrderItem) orderItem);
                return;
            case PT_DOLPHIN_HOTEL:
                addDolphinAgreementOrThrow((DolphinOrderItem) orderItem);
                return;
            case PT_TRAVELLINE_HOTEL:
                addTravellineAgreementOrThrow((TravellineOrderItem) orderItem);
                return;
            case PT_BNOVO_HOTEL:
                addBNovoAgreementOrThrow((BNovoOrderItem) orderItem);
                return;
            case PT_BRONEVIK_HOTEL:
                addBronevikAgreementOrThrow((BronevikOrderItem) orderItem);
                return;
            case PT_FLIGHT:
                addAeroflotAgreementOrThrow((AeroflotOrderItem) orderItem);
                return;
            case PT_TRAIN:
                addTrainAgreementOrThrow((TrainOrderItem) orderItem);
                return;
            case PT_SUBURBAN:
                addSuburbanAgreementOrThrow((SuburbanOrderItem) orderItem);
                return;
            case PT_BUS:
                addBusAgreementOrThrow((BusOrderItem) orderItem);
                return;
            default:
                throw new RuntimeException("Unsupported order item type " + orderItem.getPublicType());
        }
    }

    @TransactionMandatory
    public <T extends HotelOrderItem & DirectHotelBillingPartnerAgreementProvider> void checkNewClientForAgreement(
            T orderItem, long newClientId, long newContractId) {
        //check clientId and contractId once again for safety
        ContractInfo contractInfo = balanceContractDictionary.findContractInfoByClientId(newClientId);
        Preconditions.checkNotNull(contractInfo, "Contract was not found for clientId: %s", newClientId);
        Preconditions.checkArgument(newClientId != orderItem.getAgreement().getFinancialClientId(),
                "The order is already on clientId: %s", newClientId);
        Preconditions.checkArgument(newContractId == contractInfo.getContractId(),
                "Expected contract %s; got: %s", contractInfo.getContractId(), newContractId);
    }

    private void addExpediaAgreementOrThrow(ExpediaOrderItem orderItem) {
        long billingClientId = expediaProperties.getBillingClientId();
        throwIfInactivePartner(orderItem.getPublicType(), billingClientId);
        log.info("Adding an agreement to order item {}; billing client id {}", orderItem.getId(), billingClientId);
        orderItem.setBillingPartnerAgreement(ExpediaBillingPartnerAgreement.builder()
                .billingClientId(billingClientId)
                .build());
    }

    private void addDolphinAgreementOrThrow(DolphinOrderItem orderItem) {
        long billingClientId = dolphinBillingProperties.getBillingClientId();
        throwIfInactivePartner(orderItem.getPublicType(), billingClientId);
        DolphinBillingPartnerAgreement agreement = DolphinBillingPartnerAgreement.builder()
                .billingClientId(billingClientId)
                .billingContractId(dolphinBillingProperties.getBillingContractId())
                .confirmRate(dolphinBillingProperties.getAgentFeePercent())
                .build();
        log.info("Adding an agreement to order item {}; billing client id {}", orderItem.getId(), billingClientId);
        orderItem.setBillingPartnerAgreement(agreement);
    }

    private void addBronevikAgreementOrThrow(BronevikOrderItem orderItem) {
        BronevikProperties.Billing billing = bronevikProperties.getBilling();
        long billingClientId = bronevikProperties.getBilling().getClientId();
        // TODO(mokosha): Вернуть строку, когда будут известны данные биллинга
//        throwIfInactivePartner(orderItem.getPublicType(), billingClientId);
        log.info("Adding an agreement to order item {}; billing client id {}", orderItem.getId(), billingClientId);
        orderItem.setBillingPartnerAgreement(BronevikBillingPartnerAgreement.builder()
                .billingClientId(billing.getClientId())
                .billingContractId(billing.getContractId())
                .confirmRate(billing.getConfirmRate())
                .build());
    }

    private void addTravellineAgreementOrThrow(TravellineOrderItem orderItem) {
        HotelDto hotel = orderItem.getItinerary().getOffer().getHotel();
        HotelAgreement hotelAgreement;
        try {
            hotelAgreement = hotelAgreementService.getAgreementForTimestamp(
                    hotel.getCode(),
                    EPartnerId.PI_TRAVELLINE,
                    Instant.now());
        } catch (Exception e) {
            if (orderItem.getTestContext() != null) {
                log.warn("TestContext is present, will use synthetic TravellineAgreement");
                hotelAgreement = HotelAgreement.newBuilder()
                        .setId(0L)
                        .setHotelId(hotel.getCode())
                        .setPartnerId(EPartnerId.PI_TRAVELLINE)
                        .setInn("")
                        .setFinancialClientId(-10000004)
                        .setFinancialContractId(3854862340340027432L)
                        .setOrderConfirmedRate("0.1")
                        .setOrderRefundedRate("0.1")
                        .setAgreementStartDate(Instant.now().toEpochMilli())
                        .setEnabled(true)
                        .setVatType(EVat.VAT_NONE)
                        .setSendEmptyOrdersReport(true)
                        .build();
            } else {
                log.error("Problem with travelline agreement, hotel id: {}", hotel.getCode(), e);
                throw e;
            }
        }
        orderItem.setAgreement(hotelAgreement);
        long billingClientId = orderItem.getAgreement().getBillingClientId();
        if (orderItem.getTestContext() == null) {
            throwIfInactivePartner(orderItem.getPublicType(), billingClientId);
        }
        log.info("Added an agreement to order item {}; billing client id {}", billingClientId, orderItem.getId());
    }

    private void addBNovoAgreementOrThrow(BNovoOrderItem orderItem) {
        String hotelId = String.valueOf(orderItem.getItinerary().getAccountId());
        HotelAgreement agreement;
        try {
            agreement = hotelAgreementService.getAgreementForTimestamp(hotelId, EPartnerId.PI_BNOVO, Instant.now());
        } catch (Exception e) {
            if (orderItem.getTestContext() != null) {
                log.warn("TestContext is present, will use synthetic BNovoAgreement");
                agreement = HotelAgreement.newBuilder()
                        .setId(0L)
                        .setHotelId(hotelId)
                        .setPartnerId(EPartnerId.PI_BNOVO)
                        .setInn("7736207543")
                        .setFinancialClientId(-10000005)
                        .setFinancialContractId(9124678324623462894L)
                        .setOrderConfirmedRate("0.1")
                        .setOrderRefundedRate("0.1")
                        .setAgreementStartDate(Instant.now().toEpochMilli())
                        .setEnabled(true)
                        .setVatType(EVat.VAT_0)
                        .setSendEmptyOrdersReport(true)
                        .build();
            } else {
                log.error("Problem with bnovo agreement, hotel id: {}", hotelId, e);
                throw e;
            }
        }
        orderItem.setAgreement(agreement);
        long billingClientId = orderItem.getAgreement().getBillingClientId();
        if (orderItem.getTestContext() == null) {
            throwIfInactivePartner(orderItem.getPublicType(), billingClientId);
        }
        log.info("Added an agreement to order item {}; billing client id {}", billingClientId, orderItem.getId());
    }

    private void addAeroflotAgreementOrThrow(AeroflotOrderItem orderItem) {
        long billingClientId = aeroflotWorkflowProperties.getBillingClientId();
        throwIfInactivePartner(orderItem.getPublicType(), billingClientId);
        log.info("Adding an agreement to order item {}; billing client id {}", orderItem.getId(), billingClientId);
        orderItem.setBillingPartnerAgreement(AeroflotBillingPartnerAgreement.builder()
                .billingClientId(billingClientId)
                .build());
    }

    private void addTrainAgreementOrThrow(TrainOrderItem orderItem) {
        long billingClientId = trainWorkflowProperties.getBillingClientId();
        throwIfInactivePartner(orderItem.getPublicType(), billingClientId);
        log.info("Adding an agreement to order item {}; billing client id {}", orderItem.getId(), billingClientId);
        orderItem.setBillingPartnerAgreement(TrainBillingPartnerAgreement.builder()
                .billingClientId(billingClientId)
                .insuranceClientId(trainWorkflowProperties.getBillingInsuranceClientId())
                .insuranceFeeCoefficient(trainWorkflowProperties.getInsuranceFeeCoefficient())
                .build());
    }

    private void addSuburbanAgreementOrThrow(SuburbanOrderItem orderItem) {
        SuburbanProviderBase provider = suburbanEnvProvider.createEnv(orderItem).createSuburbanProvider();
        long billingClientId = provider.getBillingClientId();
        throwIfInactivePartner(orderItem.getPublicType(), billingClientId);
        log.info("Adding an agreement to order item {}; billing client id {}", orderItem.getId(), billingClientId);

        orderItem.setBillingPartnerAgreement(new SuburbanBillingPartnerAgreement(billingClientId));
    }

    private void addBusAgreementOrThrow(BusOrderItem orderItem) {
        long billingClientId = busProperties.getBillingClientId();
        throwIfInactivePartner(orderItem.getPublicType(), billingClientId);
        log.info("Adding an agreement to order item {}; billing client id {}", orderItem.getId(), billingClientId);
        orderItem.setBillingPartnerAgreement(BusBillingPartnerAgreement.builder()
                .billingClientId(billingClientId)
                .build());
    }

    @TransactionMandatory
    public void ensureCanConfirmOrder(Order order) {
        for (OrderItem orderItem : order.getOrderItems()) {
            ensureCanConfirmService(orderItem);
        }
    }

    @TransactionMandatory
    public void ensureCanConfirmService(OrderItem orderItem) {
        switch (orderItem.getPublicType()) {
            case PT_EXPEDIA_HOTEL:
            case PT_DOLPHIN_HOTEL:
            case PT_TRAVELLINE_HOTEL:
            case PT_FLIGHT:
            case PT_TRAIN:
            case PT_BUS:
                BillingPartnerAgreement agreement = orderItem.getBillingPartnerAgreement();
                throwIfInactivePartner(orderItem.getPublicType(), agreement.getBillingClientId());
                return;
            default:
                throw new IllegalArgumentException("Unsupported order item type " + orderItem.getPublicType());
        }
    }

    private void throwIfInactivePartner(EServiceType serviceType, long billingClientId) {
        if (!isPartnerAgreementActive(billingClientId)) {
            throw Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Partner agreement isn't active")
                    .withAttribute("api_partner", serviceType)
                    .withAttribute("billing_partner", billingClientId)
                    .toEx();
        }
    }
}
