package ru.yandex.travel.orders.services.payments;

import java.time.Clock;
import java.time.Instant;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.payments.model.TrustTerminalForPartner;
import ru.yandex.travel.orders.services.promo.mir2020.Mir2020PromoService;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnvProvider;

@Slf4j
@RequiredArgsConstructor
public class DefaultTrustPaymentPolicy implements TrustPaymentPolicy {

    private final Clock clock;

    private final TrustPaymentPolicyProperties properties;
    private final Mir2020PromoService mir2020PromoService;
    private final SuburbanOrderItemEnvProvider suburbanEnvProvider;

    @Override
    public boolean forceThreeDs(Order order) {
        if (order instanceof HotelOrder) {
            if (properties.getHotels().isForceEnableThreeDs()) {
                return true;
            }
            if (properties.getHotels().isForceDisableThreeDs()) {
                return false;
            }
            return !bypass3ds((HotelOrder) order);
        } else {
            // for all other orders for the moment we don't config 3ds properties here
            return false;
        }
    }

    @Override
    public boolean eligibleForMir(Order order) {
        if (OrderCompatibilityUtils.isHotelOrder(order)) {
            return mir2020PromoService.checkInitialEligibility(order);
        } else {
            // mir promo is hotel only
            return false;
        }
    }

    @Override
    public boolean processThroughYt(Order order) {
        if (OrderCompatibilityUtils.isTrainOrder(order)) {
            return properties.getTrains().isProcessThroughYt();
        } else {
            return false;
        }
    }

    @Override
    public TrustTerminalForPartner terminalForPartner(Order order) {
        if (!OrderCompatibilityUtils.isSuburbanOrder(order)) {
            return null;
        }

        SuburbanOrderItem orderItem = OrderCompatibilityUtils.getSuburbanOrderItem(order);
        return suburbanEnvProvider.createEnv(orderItem).createSuburbanProvider().getTerminalForPartner();
    }

    @Override
    public InvoicePaymentFlags getInvoicePaymentFlags(Order order) {
        return InvoicePaymentFlags.builder()
                .force3ds(forceThreeDs(order))
                .useMirPromo(eligibleForMir(order))
                .processThroughYt(processThroughYt(order))
                .terminalForPartner(terminalForPartner(order))
                .build();
    }

    private boolean bypass3ds(HotelOrder order) {
        try {
            Preconditions.checkArgument(order.getOrderItems().size() == 1, "Only one hotel order allowed");
            HotelOrderItem orderItem = (HotelOrderItem) order.getOrderItems().get(0);
            ZoneId hotelZone = orderItem.getHotelItinerary().getOrderDetails().getHotelTimeZoneId();
            Preconditions.checkArgument(hotelZone != null, "Hotel zone must be present");

            Instant now = clock.instant();

            Instant minCheckinInstant = now.plus(properties.getHotels().getNoThreeDs().getDaysBeforeCheckin(),
                    ChronoUnit.DAYS);
            Instant penaltyThresholdInstant = now.plus(properties.getHotels().getNoThreeDs().getDaysBeforePenalty(),
                    ChronoUnit.DAYS);

            Instant checkInInstant = orderItem.getHotelItinerary().getOrderDetails().getCheckinDate()
                    .atStartOfDay(hotelZone).toInstant();

            boolean checkinConditionPassed = checkInInstant.isAfter(minCheckinInstant);
            boolean refundConditionPassed = orderItem.getHotelItinerary().getRefundRules()
                    .isFullyRefundableAt(penaltyThresholdInstant);

            boolean maxPermittedSumConditionPassed =
                    order.calculateTotalCost().getNumberStripped()
                            .compareTo(properties.getHotels().getNoThreeDs().getMaxPrice()) < 0;

            return checkinConditionPassed && refundConditionPassed && maxPermittedSumConditionPassed;

        } catch (Exception e) {
            log.error("Exception occurred while trying to determine " +
                    "if bypassing three ds for hotel order is allowed, falling back to false", e);
            return false;
        }
    }
}
