package ru.yandex.travel.orders.services.payments;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.orders.entities.AeroflotInvoice;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.InvoiceItemProvider;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.PendingInvoice;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.repository.InvoiceRepository;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentCreated;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentProvider;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Service
@Slf4j
@RequiredArgsConstructor
public class InvoiceFactory {
    private final AuthorizationService authorizationService;
    private final TrustPaymentPolicy trustPaymentPolicy;
    private final Environment environment;
    private final InvoiceRepository invoiceRepository;
    private final WorkflowRepository workflowRepository;
    private final WorkflowMessageSender workflowMessageSender;

    public static EPaymentProvider paymentTypeFromInvoiceType(EInvoiceType invoiceType) {
        switch (invoiceType) {
            case IT_TRUST:
                return EPaymentProvider.PP_TRUST;
            case IT_AVIA_AEROFLOT:
                return EPaymentProvider.PP_AVIA_AEROFLOT;
            default:
                return EPaymentProvider.PP_UNKNOWN;
        }
    }

    public static EInvoiceType invoiceTypeFromPaymentType(EPaymentProvider paymentType) {
        switch (paymentType) {
            case PP_TRUST:
                return EInvoiceType.IT_TRUST;
            case PP_AVIA_AEROFLOT:
                return EInvoiceType.IT_AVIA_AEROFLOT;
            default:
                return EInvoiceType.IT_UNKNOWN;
        }
    }

    public Invoice createInvoice(Order order, EInvoiceType invoiceType, PendingInvoice pendingInvoice,
                                 String source, String returnPath, String confirmationReturnPath,
                                 String paymentMethodId, Message paymentTestContext,
                                 boolean useNewCommonPaymentWebForm) {
        Invoice invoice;
        List<InvoiceItemProvider> itemProvider;
        if (pendingInvoice == null) {
            itemProvider = order.getOrderItems().stream()
                    .flatMap(oi -> oi.getFiscalItems().stream())
                    .collect(Collectors.toList());
        } else {
            itemProvider = new ArrayList<>(pendingInvoice.getPendingInvoiceItems());
        }
        var owner = authorizationService.getOrderOwner(order.getId());
        switch (invoiceType) {
            case IT_TRUST:
                InvoicePaymentFlags paymentFlags = trustPaymentPolicy.getInvoicePaymentFlags(order);
                paymentFlags = paymentFlags.toBuilder().newCommonPaymentWebForm(useNewCommonPaymentWebForm).build();
                invoice = TrustInvoice.createInvoice(order, owner, itemProvider, paymentFlags);
                if (invoice.getInvoiceItems().size() == 0) {
                    Error.with(EErrorCode.EC_ABORTED, "No new invoice items created").andThrow();
                }
                break;
            case IT_AVIA_AEROFLOT:
                invoice = AeroflotInvoice.createInvoice(order, itemProvider, owner, useNewCommonPaymentWebForm);
                break;
            default:
                throw new IllegalArgumentException("Unsupported invoice type: " + invoiceType);
        }

        // !!!WARNING!!!
        // once again ensure that the environment is not prod,
        // so it can't be override even by hand on database level
        if (!environment.acceptsProfiles("prod") && (order.isMockPayment() || paymentTestContext != null)) {
            invoice.setPaymentProfile(PaymentProfile.MOCK_PAYMENT);
        }

        if (!Strings.isNullOrEmpty(paymentMethodId)) {
            invoice.setRequestedPaymentMethodId(paymentMethodId);
        } else {
            invoice.setSource(source);
            invoice.setReturnPath(ProtoChecks.checkStringIsPresent("return url", returnPath));
            invoice.setConfirmationReturnPath(confirmationReturnPath);
        }
        if (pendingInvoice != null) {
            pendingInvoice.addAttempt(invoice);
            invoice.setExpirationDate(pendingInvoice.getExpiresAt());
        }
        invoice.setPaymentTestContext(paymentTestContext);
        invoice = invoiceRepository.saveAndFlush(invoice);

        Workflow workflow = Workflow.createWorkflowForEntity((WorkflowEntity) invoice,
                order.getWorkflow().getId());
        workflow = workflowRepository.saveAndFlush(workflow);
        workflowMessageSender.scheduleEvent(workflow.getId(), TPaymentCreated.newBuilder().build());
        return invoice;
    }
}
