package ru.yandex.travel.orders.services.payments;

import java.time.Clock;
import java.util.Map;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.orders.services.mock.MockDBTrustClient;
import ru.yandex.travel.orders.services.mock.TrustDBMockProperties;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketPassParams;
import ru.yandex.travel.orders.services.promo.mir2020.Mir2020PromoService;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnvProvider;

@Configuration
@EnableConfigurationProperties({
        TrustConnectionProperties.class,
        TrustHotelsProperties.class,
        TrustAeroflotProperties.class,
        TrustTrainProperties.class,
        TrustSuburbanProperties.class,
        TrustBusProperties.class,
        TrustDBMockProperties.class,
        TrustPaymentPolicyProperties.class
})
@RequiredArgsConstructor
public class TrustConfiguration {
    private final Environment environment;

    @Bean
    public TrustPaymentPolicy trustPaymentPolicy(TrustPaymentPolicyProperties properties,
                                                 Clock clock,
                                                 Mir2020PromoService mir2020PromoService,
                                                 SuburbanOrderItemEnvProvider suburbanEnvProvider) {
        return new DefaultTrustPaymentPolicy(clock, properties, mir2020PromoService, suburbanEnvProvider);
    }

    @Bean
    public TrustClientProvider trustClientProvider(@Qualifier("trustAhcClientWrapper") AsyncHttpClientWrapper trustAhcClientWrapper,
                                                   TrustConnectionProperties trustConnectionProperties,
                                                   TrustHotelsProperties trustHotelsProperties,
                                                   TrustAeroflotProperties trustAeroflotProperties,
                                                   TrustTrainProperties trustTrainProperties,
                                                   TrustSuburbanProperties trustSuburbanProperties,
                                                   TrustBusProperties trustBusProperties,
                                                   ObjectProvider<MockDBTrustClient> mockDBTrustClientObjectProvider) {

        Map<PaymentProfile, String> serviceTokens = Map.ofEntries(
                Map.entry(PaymentProfile.HOTEL, trustHotelsProperties.getServiceToken()),
                Map.entry(PaymentProfile.HOTEL_RUB_SETTLEMENT, trustHotelsProperties.getServiceToken()),
                Map.entry(PaymentProfile.AEROFLOT, trustAeroflotProperties.getServiceToken()),
                Map.entry(PaymentProfile.TRAIN, trustTrainProperties.getServiceToken()),
                Map.entry(PaymentProfile.SUBURBAN, trustSuburbanProperties.getServiceToken()),
                Map.entry(PaymentProfile.AEROEXPRESS, trustSuburbanProperties.getAeroexpressServiceToken()),
                Map.entry(PaymentProfile.BUS, trustBusProperties.getServiceToken())
        );

        return paymentProfile -> {
            // double-check
            if (paymentProfile == PaymentProfile.MOCK_PAYMENT && !environment.acceptsProfiles("prod")) {
                MockDBTrustClient mockDBTrustClient = mockDBTrustClientObjectProvider.getIfUnique();
                return Preconditions.checkNotNull(mockDBTrustClient,
                        "Mock trust client requested but not present. Set trust-db-mock.enabled: true");
            }
            String serviceToken = Preconditions.checkNotNull(
                    serviceTokens.get(paymentProfile), "Service token for payment profile %s not found");
            return new TrustRetryableClientWrapper(
                    new DefaultTrustClient(trustAhcClientWrapper, trustConnectionProperties,
                            serviceToken, createCreateBasketPassParams(paymentProfile))
            );
        };
    }

    private TrustCreateBasketPassParams createCreateBasketPassParams(PaymentProfile paymentProfile) {
        TrustCreateBasketPassParams result = null;
        if (paymentProfile == PaymentProfile.HOTEL_RUB_SETTLEMENT) {
            result = new TrustCreateBasketPassParams();
            result.setRubSettlement(1);
        }
        return result;
    }
}
