package ru.yandex.travel.orders.services.payments;

import java.time.Duration;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;
import javax.validation.Valid;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.validation.annotation.Validated;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;

@ConfigurationProperties(value = "trust-hotels", ignoreUnknownFields = false)
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
@Validated
public class TrustHotelsProperties {
    @NotEmpty
    private String serviceToken;

    @NotEmpty
    private String callbackUrl;
    @NotEmpty
    private String partnerId;

    private boolean allowThreeDsForce;

    @NotEmpty
    private String taxationType;

    @NotNull
    private Duration clearingDelay;
    @NotNull
    private Duration clearingRefreshTimeout;
    @NotNull
    private Duration clearingShutdownTimeout;
    @NotNull
    private Duration paymentRefreshTimeout;

    @NotNull
    private int numberOfRefreshThreads;
    @NotNull
    private int numberOfInvoicesToClear;
    @NotNull
    private int numberOfRefundRefreshThreads;

    @NotNull
    private Duration scheduleClearingRate;

    @NotNull
    private Duration defaultPaymentTimeout;

    @NotNull
    private boolean trainsNewProcessingEnabled;

    private EDisplayOrderType defaultServiceSettings;

    @NotEmpty
    @Valid
    // the key is meaningless, this structure is used for simple overriding by key rather than by index
    private Map<String, ServiceSettings> serviceSettings;

    @Data
    @Builder
    @NoArgsConstructor
    @AllArgsConstructor
    public static class ServiceSettings {
        @NotNull
        private EDisplayOrderType orderType;
        private String paymentWebFormTemplateName;
        private String commonPaymentWebForm;
    }

    @PostConstruct
    public void init() {
        Set<EDisplayOrderType> orderTypes = serviceSettings.values().stream()
                .map(s -> s.orderType)
                .collect(Collectors.toSet());
        Preconditions.checkState(serviceSettings.size() == orderTypes.size(),
                "Duplicate order types detected: %s", serviceSettings);
        Preconditions.checkState(orderTypes.contains(defaultServiceSettings),
                "The default settings order type is absent: default %s, specified %s",
                defaultServiceSettings, orderTypes);
    }

    public ServiceSettings getSettings(EDisplayOrderType orderType) {
        ServiceSettings defaultSettings = null;
        for (ServiceSettings value : serviceSettings.values()) {
            if (value.getOrderType() == orderType) {
                return value;
            }
            if (value.getOrderType() == defaultServiceSettings) {
                defaultSettings = value;
            }
        }
        Preconditions.checkNotNull(defaultSettings, "No default settings for %s", defaultServiceSettings);
        return defaultSettings;
    }
}
