package ru.yandex.travel.orders.services.payments;

import java.util.concurrent.CompletableFuture;

import lombok.RequiredArgsConstructor;

import ru.yandex.travel.commons.http.apiclient.HttpApiRetryableException;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustClearResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateOrderResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundResponse;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentMethodsResponse;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentReceiptResponse;
import ru.yandex.travel.orders.services.payments.model.TrustRefundStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustResizeRequest;
import ru.yandex.travel.orders.services.payments.model.TrustResizeResponse;
import ru.yandex.travel.orders.services.payments.model.TrustStartPaymentResponse;
import ru.yandex.travel.orders.services.payments.model.TrustStartRefundResponse;
import ru.yandex.travel.orders.services.payments.model.TrustUnholdResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateAccountRequest;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateAccountResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateTopupResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupRequest;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupStartResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupStatusResponse;
import ru.yandex.travel.workflow.exceptions.RetryableException;

/**
 * Wrap {@link ru.yandex.travel.commons.http.apiclient.HttpApiRetryableException} to orders-specific
 * {@link ru.yandex.travel.workflow.exceptions.RetryableException}
 */
@RequiredArgsConstructor
public class TrustRetryableClientWrapper implements TrustClient {
    private final TrustClient delegate;

    @Override
    public TrustCreateOrderResponse createOrder(String productId, TrustUserInfo userInfo) {
        try {
            return delegate.createOrder(productId, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustCreateBasketResponse createBasket(TrustCreateBasketRequest request, TrustUserInfo userInfo, Object testContext) {
        try {
            return delegate.createBasket(request, userInfo, testContext);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustBasketStatusResponse getBasketStatus(String purchaseToken, TrustUserInfo userInfo) {
        try {
            return delegate.getBasketStatus(purchaseToken, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustStartPaymentResponse startPayment(String purchaseToken, TrustUserInfo userInfo) {
        try {
            return delegate.startPayment(purchaseToken, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustClearResponse clear(String purchaseToken, TrustUserInfo userInfo) {
        try {
            return delegate.clear(purchaseToken, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustUnholdResponse unhold(String purchaseToken, TrustUserInfo userInfo) {
        try {
            return delegate.unhold(purchaseToken, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustResizeResponse resize(String purchaseToken, String orderId, TrustResizeRequest resizeRequest, TrustUserInfo userInfo) {
        try {
            return delegate.resize(purchaseToken, orderId, resizeRequest, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustCreateRefundResponse createRefund(TrustCreateRefundRequest request, TrustUserInfo userInfo) {
        try {
            return delegate.createRefund(request, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustStartRefundResponse startRefund(String refundId, TrustUserInfo userInfo) {
        try {
            return delegate.startRefund(refundId, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustRefundStatusResponse getRefundStatus(String refundId, TrustUserInfo userInfo) {
        try {
            return delegate.getRefundStatus(refundId, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustPaymentMethodsResponse getPaymentMethods(TrustUserInfo userInfo) {
        try {
            return delegate.getPaymentMethods(userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustCreateAccountResponse createAccount(TrustCreateAccountRequest request, TrustUserInfo userInfo) {
        try {
            return delegate.createAccount(request, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustCreateTopupResponse createTopup(TrustTopupRequest request, TrustUserInfo userInfo) {
        try {
            return delegate.createTopup(request, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustTopupStatusResponse getTopupStatus(String purchaseToken, TrustUserInfo userInfo) {
        try {
            return delegate.getTopupStatus(purchaseToken, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustTopupStartResponse startTopup(String purchaseToken, TrustUserInfo userInfo) {
        try {
            return delegate.startTopup(purchaseToken, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public TrustPaymentReceiptResponse getReceipt(String purchaseToken, String receiptId, TrustUserInfo userInfo) {
        try {
            return delegate.getReceipt(purchaseToken, userInfo);
        } catch (HttpApiRetryableException e) {
            throw new RetryableException(e);
        }
    }

    @Override
    public CompletableFuture<TrustPaymentMethodsResponse> getPaymentMethodsAsync(TrustUserInfo userInfo) {
        return delegate.getPaymentMethodsAsync(userInfo)
                .handle((res, t) -> {
                    if (t != null) {
                        if (t instanceof HttpApiRetryableException) {
                            return CompletableFuture.<TrustPaymentMethodsResponse>failedFuture(new RetryableException(t));
                        }
                        return CompletableFuture.<TrustPaymentMethodsResponse>failedFuture(t);
                    }
                    return CompletableFuture.completedFuture(res);
                })
                .thenCompose(x -> x);
    }
}
