package ru.yandex.travel.orders.services.payments.schedule;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.promo.mir.MirEligibilityService;
import ru.yandex.travel.hotels.common.schedule.OfferPaymentSchedule;
import ru.yandex.travel.hotels.common.schedule.PaymentScheduleBuilder;
import ru.yandex.travel.hotels.common.schedule.PaymentScheduleHotelList;
import ru.yandex.travel.hotels.common.schedule.PaymentScheduleProperties;
import ru.yandex.travel.hotels.proto.EMirEligibility;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;

@RequiredArgsConstructor
public class HotelPaymentScheduleBuilder implements OrderItemPaymentScheduleBuilder {
    private final HotelPaymentScheduleBuilderProperties properties;
    private final MirEligibilityService mirEligibilityService;
    private final PaymentScheduleHotelList paymentScheduleHotelList;


    @Override
    public boolean canDeferPayment(OrderItem orderItem) {
        OfferPaymentSchedule schedule = getPaymentSchedule(orderItem);
        return schedule != null;
    }

    @Override
    public PaymentScheduleBuilderRules build(OrderItem orderItem) {
        OfferPaymentSchedule schedule = getPaymentSchedule(orderItem);
        if (schedule == null) {
            return PaymentScheduleBuilderRules.builder()
                    .initialPaymentRule(PaymentScheduleBuilderRule.builder()
                            .name("100% предоплата")
                            .orderItem(orderItem)
                            .ratio(BigDecimal.ONE)
                            .build())
                    .build();
        } else {
            return PaymentScheduleBuilderRules.builder()
                    .initialPaymentRule(PaymentScheduleBuilderRule.builder()
                            .name(schedule.getInitialPayment().getName())
                            .orderItem(orderItem)
                            .ratio(schedule.getInitialPayment().getRatio())
                            .build())
                    .deferredRules(
                            schedule.getDeferredPayments().stream().
                                    map(dp -> PaymentScheduleBuilderRule.builder()
                                            .ratio(dp.getRatio())
                                            .name(dp.getName())
                                            .paymentEndsAt(dp.getPaymentEndsAt())
                                            .emailReminderAt(dp.getPaymentEndsAt().minus(properties.getEmailInterval()))
                                            .ticketReminderAt(dp.getPaymentEndsAt().minus(properties.getTicketInterval()))
                                            .penaltyIfUnpaid(dp.getPenaltyIfUnpaid())
                                            .orderItem(orderItem)
                                            .build())
                                    .collect(Collectors.toList()))
                    .build();
        }
    }

    private OfferPaymentSchedule getPaymentSchedule(OrderItem orderItem) {
        Preconditions.checkArgument(orderItem instanceof HotelOrderItem);
        HotelOrderItem hotelOrderItem = (HotelOrderItem) orderItem;
        if (!properties.checkEnabledForPartner(hotelOrderItem.getPartnerId()) ||
                !paymentScheduleHotelList.checkPaymentScheduleIsAllowedForHotel(hotelOrderItem.getPartnerId(),
                        hotelOrderItem.getHotelItinerary().getOrderDetails().getOriginalId())) {
            return null;
        }

        HotelItinerary itinerary = hotelOrderItem.getHotelItinerary();
        var mirPromoStatus = mirEligibilityService.checkEligibility(hotelOrderItem.getPartnerId(),
                itinerary.getOrderDetails().getOriginalId(),
                Instant.now(), itinerary.getOrderDetails().getCheckinDate(),
                itinerary.getOrderDetails().getCheckoutDate(),
                orderItem.getOrder().calculateTotalCost().getNumberStripped());
        if (mirPromoStatus.getEligibility() == EMirEligibility.ME_ELIGIBLE) {
            return null;
        }

        ZoneId zoneId = itinerary.getOrderDetails().getHotelTimeZoneId();
        if (zoneId == null) {
            zoneId = ZoneId.systemDefault();
        }
        Instant checkinDayStarts = itinerary.getOrderDetails().getCheckinDate()
                .atStartOfDay(zoneId).toInstant();
        Duration paymentInterval = properties.getMinPaymentInterval();
        Duration refundSafetyWindow = properties.getRefundSafetyWindow();
        if (orderItem.getTestContext() != null && orderItem.getTestContext() instanceof THotelTestContext) {
            THotelTestContext testContext = (THotelTestContext) orderItem.getTestContext();
            if (testContext.getIgnorePaymentScheduleRestrictions()) {
                paymentInterval = PaymentScheduleProperties.MIN_PAYMENT_INTERVAL;
                refundSafetyWindow = PaymentScheduleProperties.MIN_SAFETY_WINDOW;
            }
        }

        Money minFirstPayment;
        if (properties.getMinFirstPaymentRate() != null) {
            minFirstPayment = orderItem.preliminaryTotalCost().multiply(properties.getMinFirstPaymentRate());
        } else if (properties.getMinFirstPayment() != null) {
            minFirstPayment = Money.of(properties.getMinFirstPayment(), ProtoCurrencyUnit.RUB);
        } else {
            throw new IllegalStateException("Incorrect configuration of minFirstPayment");
        }
        Money maxFirstPayment = orderItem.preliminaryTotalCost().multiply(properties.getMaxFirstPaymentRate());

        return PaymentScheduleBuilder.twoPaymentSchedule(
                itinerary.getGeneratedAtInstant(),
                checkinDayStarts, orderItem.calculateTotalCostWithPreliminaryFallback(), minFirstPayment,
                maxFirstPayment,
                itinerary.getRefundRules(), paymentInterval, refundSafetyWindow);
    }


    @Override
    public boolean appliesTo(OrderItem orderItem) {
        return orderItem instanceof HotelOrderItem;
    }

}
