package ru.yandex.travel.orders.services.payments.schedule;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.money.Monetary;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.PaymentScheduleItem;
import ru.yandex.travel.orders.entities.PendingInvoice;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Service
@RequiredArgsConstructor
public class PaymentBuilder implements InitializingBean, ApplicationContextAware {

    private final WorkflowRepository workflowRepository;
    private ApplicationContext applicationContext;
    private List<OrderItemPaymentScheduleBuilder> builders;

    public boolean canDeferPaymentForOrder(Order order) {
        return order.getOrderItems().stream().anyMatch(this::canDeferPaymentForOrderItem);
    }

    public Payment buildDeferred(Order order) {
        List<PaymentScheduleBuilderRules> rulesByOrderItem =
                order.getOrderItems().stream().map(this::buildRulesForOrderItem).collect(Collectors.toList());

        List<PaymentScheduleBuilderRule> deferredRules =
                rulesByOrderItem.stream().flatMap(rules -> rules.getDeferredRules().stream())
                        .sorted(Comparator.comparing(PaymentScheduleBuilderRule::getPaymentEndsAt))
                        .collect(Collectors.toList());

        if (deferredRules.size() == 0) {
            return null;
        }

        PaymentSchedule schedule = PaymentSchedule.builder()
                .id(UUID.randomUUID())
                .order(order)
                .expired(false)
                .items(new ArrayList<>())
                .state(EPaymentState.PS_DRAFT)
                .build();
        schedule.setInitialPendingInvoice(PendingInvoice.builder()
                .id(UUID.randomUUID())
                .paymentSchedule(schedule)
                .expiresAt(order.getExpiresAt())
                .state(EPaymentState.PS_DRAFT)
                .build());

        Workflow scheduleWorkflow = Workflow.createWorkflowForEntity(schedule, order.getWorkflow().getId());
        workflowRepository.save(scheduleWorkflow);
        schedule.setWorkflow(scheduleWorkflow);

        Workflow initialInvoiceWorkflow = Workflow.createWorkflowForEntity(schedule.getInitialPendingInvoice(),
                scheduleWorkflow.getId());
        workflowRepository.save(initialInvoiceWorkflow);
        schedule.getInitialPendingInvoice().setWorkflow(initialInvoiceWorkflow);

        rulesByOrderItem.stream().map(PaymentScheduleBuilderRules::getInitialPaymentRule).forEach(rule -> applyRuleToInvoice(rule, schedule.getInitialPendingInvoice()));


        PaymentScheduleItem scheduleItem = null;
        PaymentScheduleBuilderRule prevRule = null;
        for (var rule : deferredRules) {
            if (prevRule == null || !prevRule.getPaymentEndsAt().equals(rule.getPaymentEndsAt())) {
                if (scheduleItem != null) {
                    schedule.getItems().add(scheduleItem);
                }
                scheduleItem = PaymentScheduleItem.builder()
                        .id(UUID.randomUUID())
                        .name(rule.getName())
                        .schedule(schedule)
                        .reminderEmailSent(false)
                        .reminderTicket(null)
                        .ratio(rule.getRatio())
                        .emailReminderAt(rule.getEmailReminderAt())
                        .ticketReminderAt(rule.getTicketReminderAt())
                        .paymentEndsAt(rule.getPaymentEndsAt())
                        .penaltyIfUnpaid(rule.getPenaltyIfUnpaid())
                        .build();
                scheduleItem.setPendingInvoice(PendingInvoice.builder()
                        .id(UUID.randomUUID())
                        .paymentSchedule(schedule)
                        .state(EPaymentState.PS_DRAFT)
                        .build());

                Workflow pendingDeferredInvoiceWorkflow =
                        Workflow.createWorkflowForEntity(scheduleItem.getPendingInvoice(),
                                scheduleWorkflow.getId());
                workflowRepository.save(pendingDeferredInvoiceWorkflow);
                scheduleItem.getPendingInvoice().setWorkflow(pendingDeferredInvoiceWorkflow);
            } else {
                scheduleItem.setName(scheduleItem.getName() + ", " + rule.getName());
            }
            applyRuleToInvoice(rule, scheduleItem.getPendingInvoice());
            prevRule = rule;
        }
        workflowRepository.flush();
        schedule.getItems().add(scheduleItem);
        order.setPaymentSchedule(schedule);
        return schedule;
    }

    public Payment buildImmediate(Order order) {
        var invoice = PendingInvoice.builder()
                .id(UUID.randomUUID())
                .order(order)
                .state(EPaymentState.PS_DRAFT)
                .expiresAt(order.getExpiresAt())
                .build();
        order.getOrderItems().stream()
                .flatMap(oi -> oi.getFiscalItems().stream())
                .forEach(invoice::addFullItemForFiscalItem);
        order.addPendingInvoice(invoice);
        Workflow pendingInvoiceWorkflow =
                Workflow.createWorkflowForEntity(invoice, order.getWorkflow().getId());
        workflowRepository.save(pendingInvoiceWorkflow);
        return invoice;
    }

    private void applyRuleToInvoice(PaymentScheduleBuilderRule rule, PendingInvoice invoice) {
        for (FiscalItem fi : rule.getOrderItem().getFiscalItems()) {
            Preconditions.checkArgument(fi.getYandexPlusToWithdraw().isZero(),
                    "Yandex Plus points aren't supported in deferred payments but gut %s to withdraw",
                    fi.getYandexPlusToWithdraw());
            Money nextPayment = fi.getMoneyAmount().multiply(rule.getRatio()).with(Monetary.getDefaultRounding());
            invoice.addItemForFiscalItem(fi, nextPayment, Money.zero(nextPayment.getCurrency()));
        }
    }

    private boolean canDeferPaymentForOrderItem(OrderItem item) {
        return builders.stream().anyMatch(builder -> builder.appliesTo(item) && builder.canDeferPayment(item));
    }

    private PaymentScheduleBuilderRules buildRulesForOrderItem(OrderItem item) {
        for (var builder : builders) {
            if (builder.appliesTo(item)) {
                return builder.build(item);
            }
        }
        return PaymentScheduleBuilderRules.builder()
                .initialPaymentRule(PaymentScheduleBuilderRule.builder()
                        .ratio(BigDecimal.ONE)
                        .orderItem(item)
                        .build())
                .build();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        builders = new ArrayList<>(applicationContext.getBeansOfType(OrderItemPaymentScheduleBuilder.class).values());
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
