package ru.yandex.travel.orders.services.pdfgenerator;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfGenerateBusesTicketsRequest;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfGenerateHotelsBusinessTripDocRequest;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfGenerateHotelsVoucherRequest;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfStateResponse;

@Slf4j
public class PdfGeneratorServiceImpl implements PdfGeneratorService {
    private final AsyncHttpClientWrapper client;
    private final PdfGeneratorConfigurationProperties config;
    private final ObjectMapper objectMapper;

    public PdfGeneratorServiceImpl(AsyncHttpClientWrapper client, PdfGeneratorConfigurationProperties config) {
        this.client = client;
        this.config = config;
        this.objectMapper = createObjectMapper();
    }

    public static ObjectMapper createObjectMapper() {
        return new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
                .setPropertyNamingStrategy(PropertyNamingStrategy.LOWER_CAMEL_CASE)
                .registerModule(new JavaTimeModule())
                .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
    }

    private CompletableFuture<byte[]> downloadDocumentAsBytesAsync(String documentUrl) {
        RequestBuilder builder = new RequestBuilder()
                .setMethod("GET")
                .setRequestTimeout(Math.toIntExact(config.getQuickRequestTimeout().toMillis()))
                .setReadTimeout(Math.toIntExact(config.getQuickReadTimeout().toMillis()))
                .setUrl(documentUrl)
                .setFollowRedirect(true);
        return client.executeRequest(builder, Method.PDF_DOWNLOAD.toString())
                .thenApply(response -> {
                    if (response.getStatusCode() == 200) {
                        return response.getResponseBodyAsBytes();
                    } else if (response.getStatusCode() >= 500) {
                        throw new PdfGeneratorRetryableException(httpErrorFormat(response));
                    } else if (response.getStatusCode() == 404) {
                        throw new PdfNotFoundException(httpErrorFormat(response));
                    } else {
                        throw new RuntimeException(httpErrorFormat(response));
                    }
                });
    }


    private CompletableFuture<Void> generateBusesTicketsAsync(PdfGenerateBusesTicketsRequest request) {
        return callGenerateAsync(Method.PDF_GENERATE_BUSES_TICKETS, request);
    }

    private CompletableFuture<Void> generateHotelsVoucherAsync(PdfGenerateHotelsVoucherRequest request) {
        return callGenerateAsync(Method.PDF_GENERATE_HOTELS_VOUCHER, request);
    }

    private CompletableFuture<Void> generateHotelsBusinessTripDocAsync(PdfGenerateHotelsBusinessTripDocRequest request) {
        return callGenerateAsync(Method.PDF_GENERATE_HOTELS_BUSINESS_TRIP_DOC, request);
    }

    @Override
    public void generateHotelsVoucher(PdfGenerateHotelsVoucherRequest request) {
        sync(generateHotelsVoucherAsync(request));
    }

    @Override
    public void generateHotelsBusinessTripDoc(PdfGenerateHotelsBusinessTripDocRequest request) {
        sync(generateHotelsBusinessTripDocAsync(request));
    }

    private String httpErrorFormat(Response response) {
        return String.format("HTTP error %s: %s for uri %s. Detail: %s",
                response.getStatusCode(), response.getStatusText(), response.getUri(), response.getResponseBody());
    }

    private CompletableFuture<PdfStateResponse> getStateAsync(String fileName) {
        Method method = Method.PDF_GET_STATE;
        RequestBuilder requestBuilder = new RequestBuilder()
                .setMethod("GET")
                .setUrl(config.getBaseUrl() + method.getPath())
                .setRequestTimeout(Math.toIntExact(config.getQuickRequestTimeout().toMillis()))
                .setReadTimeout(Math.toIntExact(config.getQuickReadTimeout().toMillis()))
                .addQueryParam("fileName", fileName);
        return client.executeRequest(requestBuilder, method.toString())
                .thenApply(response -> {
                    if (response.getStatusCode() == 200) {
                        try {
                            return objectMapper.readValue(response.getResponseBody(), PdfStateResponse.class);
                        } catch (IOException e) {
                            throw new RuntimeException("Couldn't deserialize response from DocumentGenerator");
                        }
                    } else if (response.getStatusCode() >= 500) {
                        throw new PdfGeneratorRetryableException(httpErrorFormat(response));
                    } else if (response.getStatusCode() == 404) {
                        throw new PdfNotFoundException(httpErrorFormat(response));
                    } else {
                        throw new RuntimeException(httpErrorFormat(response));
                    }
                });
    }

    @Override
    public byte[] downloadDocumentAsBytesSync(String documentUrl) {
        return sync(downloadDocumentAsBytesAsync(documentUrl));
    }

    @Override
    public void generateBusesTickets(PdfGenerateBusesTicketsRequest request) {
        sync(generateBusesTicketsAsync(request));
    }

    @Override
    public PdfStateResponse getState(String fileName) {
        return sync(getStateAsync(fileName));
    }

    private <T> CompletableFuture<Void> callGenerateAsync(Method method, T request) {
        RequestBuilder requestBuilder = new RequestBuilder()
                .setMethod("POST")
                .setUrl(config.getBaseUrl() + method.getPath())
                .setHeader("Content-Type", "application/json")
                .setRequestTimeout(Math.toIntExact(config.getQuickRequestTimeout().toMillis()))
                .setReadTimeout(Math.toIntExact(config.getQuickReadTimeout().toMillis()))
                .setBody(serializeRequest(request));
        return client.executeRequest(requestBuilder, method.toString())
                .thenApply(response -> {
                    if (response.getStatusCode() == 200) {
                        return null;
                    } else if (response.getStatusCode() >= 500) {
                        throw new PdfGeneratorRetryableException(httpErrorFormat(response));
                    } else {
                        throw new RuntimeException(httpErrorFormat(response));
                    }
                });
    }

    private <T> T sync(CompletableFuture<T> future) {
        try {
            return future.get();
        } catch (InterruptedException e) {
            log.error("Pdf generator call interrupted", e);
            Thread.currentThread().interrupt(); // preserved interruption status
            throw new PdfGeneratorRetryableException(e);
        } catch (ExecutionException e) {
            if (e.getCause() == null) {
                throw new RuntimeException("No root cause for ExecutionException found", e);
            }
            Throwable cause = e.getCause();
            if (cause instanceof PdfGeneratorRetryableException) {
                throw (PdfGeneratorRetryableException) cause;
            } else if (cause instanceof TimeoutException) {
                throw new PdfGeneratorRetryableException(e);
            } else if (cause instanceof IOException) {
                throw new PdfGeneratorRetryableException(e);
            } else if (cause instanceof RuntimeException) {
                throw (RuntimeException) cause;
            } else {
                throw new RuntimeException(e);
            }
        }
    }

    private <T> String serializeRequest(T request) {
        try {
            return objectMapper.writeValueAsString(request);
        } catch (JsonProcessingException e) {
            log.error("Error serializing request", e);
            throw new RuntimeException(e);
        }
    }

    @RequiredArgsConstructor
    @Getter
    public enum Method {
        PDF_GET_LINK("/generate"),
        PDF_GENERATE_BUSES_TICKETS("/generateBusesTickets"),
        PDF_GENERATE_HOTELS_VOUCHER("/generateHotelVoucher"),
        PDF_GENERATE_HOTELS_BUSINESS_TRIP_DOC("/generateHotelBusinessTripDoc"),

        PDF_GET_STATE("/state"),
        PDF_DOWNLOAD("");

        private final String path;
    }
}
