package ru.yandex.travel.orders.services.plus;

import java.math.BigDecimal;
import java.util.Objects;
import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.repository.OrderItemRepository;
import ru.yandex.travel.orders.repository.YandexPlusTopupRepository;
import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.orders.services.payments.PaymentProfile;
import ru.yandex.travel.orders.services.plus.YandexPlusInitTopupOperation.TopupData;
import ru.yandex.travel.orders.workflow.plus.proto.TTopupStart;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;
import ru.yandex.travel.workflow.single_operation.SimpleSingleOperationRunner;

@Service
@RequiredArgsConstructor
@Slf4j
public class YandexPlusInitTopupOperation extends SimpleSingleOperationRunner<TopupData, UUID> {
    public static final OperationTypes TYPE = OperationTypes.YANDEX_PLUS_INIT_TOPUP;

    private final OrderItemRepository orderItemRepository;
    private final YandexPlusTopupRepository yandexPlusTopupRepository;
    private final WorkflowRepository workflowRepository;
    private final WorkflowMessageSender workflowMessageSender;

    @Override
    public UUID runOperation(TopupData params) {
        if (params.getOrderItemId() != null) {
            OrderItem item = orderItemRepository.findById(params.getOrderItemId()).orElseThrow();
            return runWithOrder(item, params);
        } else {
            return runWithoutOrder(params);
        }
    }

    private UUID runWithOrder(OrderItem item, TopupData params) {
        Order order = item.getOrder();
        Invoice invoice = order.getCurrentInvoice();
        try (var ignored = NestedMdc.forEntity(item)) {
            if (!isStillEligible(item, params.getIgnoreOrderStatus())) {
                log.info("Can not apply PLUS top-up");
                return null;
            }
            YandexPlusTopup topup = new YandexPlusTopup();
            topup.setId(UUID.randomUUID());
            topup.setOrderItem(item);
            topup.setAmount(params.getPoints());
            topup.setCurrency(params.getCurrency());
            topup.setPaymentProfile(invoice.getPaymentProfile());
            if (Strings.isEmpty(params.passportId)) {
                topup.setPassportId(invoice.getPassportId());
            } else {
                topup.setPassportId(params.passportId);
            }
            topup.setUserIp(order.getIp());
            return runPreparedTopup(topup);
        }
    }

    private UUID runWithoutOrder(TopupData params) {
        YandexPlusTopup topup = new YandexPlusTopup();
        topup.setId(UUID.randomUUID());

        Preconditions.checkState(params.getTopupDataForExternalOrder() != null);
        topup.setExternalOrderId(params.getTopupDataForExternalOrder().getExternalOrderId());
        topup.setTotalAmountForPayload(params.getTopupDataForExternalOrder().getTotalAmountForPayload());
        topup.setCommissionAmountForPayload(params.getTopupDataForExternalOrder().getCommissionAmountForPayload());
        topup.setVatCommissionAmountForPayload(params.getTopupDataForExternalOrder().getVatCommissionAmountForPayload());

        topup.setAmount(params.getPoints());
        topup.setCurrency(params.getCurrency());
        topup.setPaymentProfile(params.getTopupDataForExternalOrder().getPaymentProfile());
        topup.setPassportId(params.getPassportId());
        topup.setUserIp(params.getTopupDataForExternalOrder().getUserIp());
        topup.setSkipFinancialEvents(params.getTopupDataForExternalOrder().isSkipFinancialEvents());

        return runPreparedTopup(topup);
    }

    private UUID runPreparedTopup(YandexPlusTopup topup) {
        topup = yandexPlusTopupRepository.saveAndFlush(topup);
        Workflow workflow = Workflow.createWorkflowForEntity(topup,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.saveAndFlush(workflow);
        workflowMessageSender.scheduleEvent(workflow.getId(), TTopupStart.newBuilder().build());
        return topup.getId();
    }

    private boolean isStillEligible(OrderItem orderItem, Boolean ignoreOrderStatus) {
        if (orderItem instanceof HotelOrderItem) {
            var hotelItem = (HotelOrderItem) orderItem;
            if (Objects.requireNonNullElse(ignoreOrderStatus, false)) {
                return true;
            }
            return hotelItem.getPublicState() == HotelOrderItem.HotelOrderItemState.CONFIRMED;
        }
        throw new RuntimeException("Can not check can apply top-up for order item type=" + orderItem.getPublicType());
    }

    @Data
    public static class TopupDataForExternalOrder {
        private String externalOrderId;
        private BigDecimal totalAmountForPayload;
        private BigDecimal commissionAmountForPayload;
        private BigDecimal vatCommissionAmountForPayload;
        private PaymentProfile paymentProfile;
        private String userIp;
        private boolean skipFinancialEvents;
    }

    @Data
    public static class TopupData {
        private ProtoCurrencyUnit currency;
        private Integer points;
        private UUID orderItemId;
        private String passportId;
        private Boolean ignoreOrderStatus;
        private TopupDataForExternalOrder topupDataForExternalOrder;
    }
}
