package ru.yandex.travel.orders.services.plus;

import java.math.BigDecimal;

import javax.money.CurrencyUnit;

import com.google.common.base.Preconditions;
import lombok.Builder;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.finances.OverallServiceBalance;
import ru.yandex.travel.orders.services.finances.providers.ProviderHelper;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupCashbackType;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupPayload;
import ru.yandex.travel.orders.workflows.plus.topup.YandexPlusPromoProperties;
import ru.yandex.travel.orders.workflows.plus.topup.YandexPlusPromoProperties.PayloadProperties;

@Service
@RequiredArgsConstructor
@Slf4j
public class YandexPlusPayloadService {
    public static final CurrencyUnit DEFAULT_CURRENCY = ProtoCurrencyUnit.RUB;

    private final YandexPlusPromoProperties properties;
    private final FinancialEventService financialEventService;

    public TrustTopupPayload createWithdrawPayloadForService(Order order) {
        Preconditions.checkArgument(order.getDisplayType() == EDisplayOrderType.DT_HOTEL,
                "Only hotel services participate in Plus cashback");
        PayloadProperties payloadParams = properties.getPayload();
        return TrustTopupPayload.builder()
                .cashbackService(payloadParams.getCashbackService())
                .serviceId(FinancialEventPaymentScheme.HOTELS.getServiceId())
                .issuer(payloadParams.getIssuer())
                .campaignName(payloadParams.getCampaignName())
                .ticket(payloadParams.getTicket())
                .build();
    }

    public TrustTopupPayload createTopupPayloadForService(OrderItem service) {
        Preconditions.checkArgument(service instanceof HotelOrderItem,
                "Only hotel services participate in Plus cashback");
        ServiceFinancialData finData = calculateFinancialDataWithTemporarySafeFallback((HotelOrderItem) service);
        return createTopupPayloadRaw(finData, properties.getPayload(), FinancialEventPaymentScheme.HOTELS);
    }

    public TrustTopupPayload createTopupPayloadForTours(ServiceFinancialData finData, FinancialEventPaymentScheme financialEventPaymentScheme) {
        Preconditions.checkArgument(financialEventPaymentScheme == FinancialEventPaymentScheme.HOTELS,
                "Only hotel services participate in Plus cashback");
        return createTopupPayloadRaw(finData, properties.getToursPayload(), financialEventPaymentScheme);
    }

    private TrustTopupPayload createTopupPayloadRaw(ServiceFinancialData finData, PayloadProperties payloadParams,
                                                    FinancialEventPaymentScheme financialEventPaymentScheme) {
        return TrustTopupPayload.builder()
                .cashbackService(payloadParams.getCashbackService())
                // only purchase based topups for plus users here, only for hotel (and tours) orders
                .cashbackType(TrustTopupCashbackType.TRANSACTION)
                .hasPlus(true)
                .baseAmount(finData.getTotal())
                .commissionAmount(finData.getFee())
                .vatCommissionAmount(finData.getFeeVat())
                .serviceId(financialEventPaymentScheme.getServiceId())
                .issuer(payloadParams.getIssuer())
                .campaignName(payloadParams.getCampaignName())
                .ticket(payloadParams.getTicket())
                .country(payloadParams.getCountry())
                .currency(payloadParams.getCurrency())
                .build();
    }

    private ServiceFinancialData calculateFinancialDataWithTemporarySafeFallback(HotelOrderItem service) {
        try {
            return calculateFinancialData(service);
        } catch (Exception e) {
            log.warn("Failed to calculate proper service financial data for topup payload; " +
                    "service id {}", service.getId(), e);
            // todo(tlg-13): the fallback should be removed when all partner types are tested & fixed (in August)
            Money total = service.getHotelItinerary().getFiscalPrice();
            return ServiceFinancialData.builder()
                    .total(total.getNumberStripped())
                    .fee(total.multiply(0.1).getNumberStripped())
                    .build();
        }
    }

    private ServiceFinancialData calculateFinancialData(HotelOrderItem service) {
        OverallServiceBalance balance = financialEventService.getOverallServiceBalance(service);
        Preconditions.checkArgument(balance.getCurrency().getCurrencyCode().equals(DEFAULT_CURRENCY.getCurrencyCode()),
                "Unsupported services currency - %s, only %s is supported at the moment",
                balance.getCurrency(), DEFAULT_CURRENCY);
        BigDecimal feeVatValue = getFeeVatValue(service, balance.getFee());
        return ServiceFinancialData.builder()
                .total(balance.getTotal().getNumberStripped())
                .fee(balance.getFee().getNumberStripped())
                .feeVat(feeVatValue)
                .build();
    }

    private BigDecimal getFeeVatValue(HotelOrderItem service, Money fee) {
        try {
            VatType vat = financialEventService.getYandexFeeVat(service);
            Money feeVatValue = fee.multiply(vat.getPercent()).divide(vat.getBasePercentWithFallback());
            return ProviderHelper.roundMoney(feeVatValue.getNumberStripped());
        } catch (Exception e) {
            log.warn("Failed to calculate fee vat value for service {}, fee amount - {}", service.getId(), fee, e);
            // accounting analytics will apply some default percent (20%)
            return null;
        }
    }

    @Value
    @Builder
    public static class ServiceFinancialData {
        private final BigDecimal total;
        private final BigDecimal fee;
        private final BigDecimal feeVat;
    }
}
