package ru.yandex.travel.orders.services.plus;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Optional;
import java.util.UUID;

import javax.annotation.Nullable;

import com.google.protobuf.Int32Value;
import com.google.protobuf.Timestamp;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.lang.ComparatorUtils;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.orders.promo.AppliedPromoCampaigns;
import ru.yandex.travel.hotels.common.orders.promo.YandexPlusApplication;
import ru.yandex.travel.orders.admin.proto.EYandexPlusMode;
import ru.yandex.travel.orders.admin.proto.TYandexPlusInfo;
import ru.yandex.travel.orders.cpa.EYandexPlusCpaMode;
import ru.yandex.travel.orders.cpa.TYandexPlusCpaInfo;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.repository.YandexPlusTopupRepository;
import ru.yandex.travel.orders.workflow.plus.proto.TTopupInfo;
import ru.yandex.travel.orders.workflow.plus.proto.TWithdrawalInfo;
import ru.yandex.travel.orders.workflows.plus.topup.YandexPlusPromoProperties;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;
import ru.yandex.travel.workflow.single_operation.proto.ESingleOperationState;

import static ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils.getOnlyHotelOrderItem;


@Service
@RequiredArgsConstructor
@Slf4j
public class YandexPlusPromoService {
    private final YandexPlusPromoProperties yandexPlusPromoProperties;
    private final SingleOperationService singleOperationService;
    private final YandexPlusTopupRepository topupRepository;

    @Nullable
    public TYandexPlusInfo getYandexPlusInfoForOrder(HotelOrder order) {
        HotelOrderItem orderItem = getOnlyHotelOrderItem(order);
        return getYandexPlusApplicationFromHotelOrderItem(orderItem)
                .map(ypa -> {
                    switch (ypa.getMode()) {
                        case TOPUP:
                            return getTopupInfo(orderItem, ypa);
                        case WITHDRAW:
                            return getWithdrawalInfo(orderItem, ypa);
                        default:
                            return null;
                    }
                })
                .orElse(null);
    }

    @Nullable
    public TYandexPlusCpaInfo getYandexPlusCpaInfoForOrder(HotelOrder order) {
        HotelOrderItem orderItem = getOnlyHotelOrderItem(order);
        return getYandexPlusApplicationFromHotelOrderItem(orderItem)
                .map(ypa -> {
                    switch (ypa.getMode()) {
                        case TOPUP:
                            return getTopupCpaInfo(orderItem, ypa);
                        case WITHDRAW:
                            return getWithdrawalCpaInfo(orderItem, ypa);
                        default:
                            return null;
                    }
                })
                .orElse(null);
    }

    private Optional<YandexPlusApplication> getYandexPlusApplicationFromHotelOrderItem(HotelOrderItem orderItem) {
        return Optional.ofNullable(orderItem.getHotelItinerary())
                .map(HotelItinerary::getAppliedPromoCampaigns)
                .map(AppliedPromoCampaigns::getYandexPlus);
    }

    private TYandexPlusCpaInfo.Builder setUserBalance(TYandexPlusCpaInfo.Builder builder,
                                                      HotelOrderItem orderItem) {
        HotelItinerary itinerary = orderItem.getHotelItinerary();
        if (itinerary.getUserInfo() != null && itinerary.getUserInfo().getPlusBalance() != null) {
            builder.setUserBalance(Int32Value.newBuilder()
                    .setValue(itinerary.getUserInfo().getPlusBalance())
                    .build());
        }
        return builder;
    }

    private TYandexPlusInfo getWithdrawalInfo(HotelOrderItem orderItem,
                                              YandexPlusApplication yandexPlusApplication) {
        if (orderItem.getPublicState() == HotelOrderItem.HotelOrderItemState.CANCELLED) {
            return null;
        }
        TYandexPlusInfo.Builder builder = TYandexPlusInfo.newBuilder();
        builder.setMode(EYandexPlusMode.YPM_WITHDRAWAL);
        builder.setWithdrawalInfo(TWithdrawalInfo.newBuilder()
                .setAmount(yandexPlusApplication.getPoints()));
        return builder.build();
    }

    private TYandexPlusCpaInfo getWithdrawalCpaInfo(HotelOrderItem orderItem,
                                                    YandexPlusApplication yandexPlusApplication) {
        if (orderItem.getPublicState() == HotelOrderItem.HotelOrderItemState.CANCELLED) {
            return null;
        }
        return setUserBalance(TYandexPlusCpaInfo.newBuilder(), orderItem)
                .setMode(EYandexPlusCpaMode.YPCM_WITHDRAWAL)
                .setWithdrawPoints(yandexPlusApplication.getPoints())
                .build();
    }

    private TYandexPlusInfo getTopupInfo(HotelOrderItem orderItem, YandexPlusApplication yandexPlusApplication) {
        TYandexPlusInfo.Builder builder = TYandexPlusInfo.newBuilder();
        builder.setMode(EYandexPlusMode.YPM_TOPUP);

        TTopupInfo.Builder topupDTO = TTopupInfo.newBuilder();
        topupDTO.setAmount(yandexPlusApplication.getPoints());

        YandexPlusTopup topupEntity = topupRepository.findByOrderItemId(orderItem.getId());
        if (topupEntity != null) {
            topupDTO.setState(topupEntity.getState());
            // TODO clarify if there's really no other way to get the top up date, without re-calculating it
            topupDTO.setTopupDate(
                    ProtoUtils.fromInstant(
                            Optional.ofNullable(topupEntity.getAuthorizedAt())
                                    .orElseGet(() -> getTopupAt(orderItem, false))
                    )
            );
            Optional.ofNullable(topupEntity.getPurchaseToken()).ifPresent(topupDTO::setPurchaseToken);
        } else {
            topupDTO.setTopupDate(ProtoUtils.fromInstant(getTopupAt(orderItem, false)));
        }
        builder.setTopupInfo(topupDTO);
        return builder.build();
    }

    private TYandexPlusCpaInfo getTopupCpaInfo(HotelOrderItem orderItem, YandexPlusApplication yandexPlusApplication) {
        YandexPlusTopup topupEntity = topupRepository.findByOrderItemId(orderItem.getId());
        Timestamp topupDate = ProtoUtils.fromInstant(
                (topupEntity != null && topupEntity.getAuthorizedAt() != null) ?
                        topupEntity.getAuthorizedAt() :
                        getTopupAt(orderItem, false));

        return setUserBalance(TYandexPlusCpaInfo.newBuilder(), orderItem)
                .setMode(EYandexPlusCpaMode.YPCM_TOPUP)
                .setTopupPoints(yandexPlusApplication.getPoints())
                .setTopupDate(topupDate)
                .build();
    }

    public void registerConfirmedService(HotelOrderItem orderItem) {
        Optional<YandexPlusApplication> yandexPlus = getYandexPlusApplicationFromHotelOrderItem(orderItem);
        if (yandexPlus.isEmpty()) {
            log.info("The service is NOT eligible for Yandex Plus");
            return;
        }
        YandexPlusApplication ypa = yandexPlus.get();
        switch (ypa.getMode()) {
            case TOPUP:
                Instant topupAt = getTopupAt(orderItem, true);
                log.info("The service is ELIGIBLE for Yandex Plus points topup: {} points on {}", ypa, topupAt);
                scheduleTopupOperationForOrder(ypa.getPoints(), orderItem.getOrder().getCurrency(),
                        orderItem.getId(), null, topupAt, false);
                return;
            case WITHDRAW:
                log.info("The service has been partially PAID with {} Yandex Plus points", ypa.getPoints());
                return;
            default:
                throw new IllegalArgumentException("Unsupported Yandex Plus application mode: " + ypa.getMode());
        }
    }

    public UUID scheduleTopupOperationForOrder(int points, ProtoCurrencyUnit currency, UUID orderItemId,
                                               String passportUid, Instant scheduleAt, boolean ignoreOrderStatus) {
        var opData = new YandexPlusInitTopupOperation.TopupData();
        opData.setPoints(points);
        opData.setCurrency(currency);
        opData.setOrderItemId(orderItemId);
        opData.setPassportId(passportUid);
        opData.setIgnoreOrderStatus(ignoreOrderStatus);
        return scheduleTopupOperation(opData, scheduleAt, "YandexPlusInitTopup:" + orderItemId, false);
    }

    public UUID scheduleTopupOperation(YandexPlusInitTopupOperation.TopupData topupData, Instant scheduleAt,
                                       String operationName, boolean unique) {
        if (scheduleAt == null) {
            scheduleAt = Instant.now();
        }

        UUID topupOperationId;
        if (unique) {
            topupOperationId = singleOperationService.scheduleUniqueOperation(operationName,
                    YandexPlusInitTopupOperation.TYPE.getValue(), topupData, scheduleAt);
        } else {
            topupOperationId = singleOperationService.scheduleOperation(operationName,
                    YandexPlusInitTopupOperation.TYPE.getValue(), topupData, scheduleAt);
        }
        log.info("Topup operation id: {}", topupOperationId);
        return topupOperationId;
    }

    public ScheduledTopupOperationInfo tryGetTopupOperationInfo(String name) {
        if (!singleOperationService.hasOperationsWithName(name)) {
            return null;
        }
        var operation = singleOperationService.getSingleOperationByUniqueName(name);
        var scheduledTopupOperationInfo = new ScheduledTopupOperationInfo();
        scheduledTopupOperationInfo.setSingleOperationState(operation.getState());
        scheduledTopupOperationInfo.setScheduledAt(operation.getScheduledAt());
        if (operation.getState() == ESingleOperationState.ERS_SUCCESS) {
            var result = singleOperationService.getOperationResultByUniqueName(name, UUID.class);
            scheduledTopupOperationInfo.setTopupEntityId(result);
        }
        return scheduledTopupOperationInfo;
    }

    public Instant getTopupAt(HotelOrderItem orderItem, boolean withLogs) {
        OrderDetails orderDetails = orderItem.getHotelItinerary().getOrderDetails();
        LocalDate checkOut = orderDetails.getCheckoutDate();
        ZoneId hotelTimeZoneId = orderDetails.getHotelTimeZoneId();
        if (hotelTimeZoneId == null) {
            hotelTimeZoneId = yandexPlusPromoProperties.getDefaultHotelTimeZoneId();
        }
        Instant checkoutDayStart = checkOut.atStartOfDay(hotelTimeZoneId).toInstant();
        if (!yandexPlusPromoProperties.getTopupDelay().equals(Duration.ZERO) && withLogs) {
            log.warn("Yandex Plus topupAt normally happen on {} but we're shifting it for {}",
                    checkoutDayStart, yandexPlusPromoProperties.getTopupDelay());
        }
        Instant topupAt = checkoutDayStart.plus(yandexPlusPromoProperties.getTopupDelay());
        if (ComparatorUtils.isLessThan(topupAt, yandexPlusPromoProperties.getMinTopupAt())) {
            if (withLogs) {
                log.warn("Yandex Plus financial events aren't ready yet, moving the topup date a bit: {} -> {}",
                        topupAt, yandexPlusPromoProperties.getMinTopupAt());
            }
            topupAt = yandexPlusPromoProperties.getMinTopupAt();
        }
        return topupAt;
    }

    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class ScheduledTopupOperationInfo {
        ESingleOperationState singleOperationState;
        Instant scheduledAt;
        UUID topupEntityId;
    }
}
