package ru.yandex.travel.orders.services.promo;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.lang.MoneyUtils;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.promo.DiscountApplicationConfig;
import ru.yandex.travel.orders.entities.promo.HotelRestriction;
import ru.yandex.travel.orders.entities.promo.PromoCode;
import ru.yandex.travel.tx.utils.TransactionMandatory;

import static ru.yandex.travel.orders.workflows.order.OrderUtils.mapServiceTypeToDisplayOrderType;


@Service
@RequiredArgsConstructor
public class DefaultPromoDiscountCalculator implements PromoCodeDiscountCalculator {

    private static final Set<EServiceType> HOTEL_SERVICE_TYPES = Set.of(
            EServiceType.PT_BNOVO_HOTEL, EServiceType.PT_DOLPHIN_HOTEL,
            EServiceType.PT_EXPEDIA_HOTEL, EServiceType.PT_TRAVELLINE_HOTEL,
            EServiceType.PT_BRONEVIK_HOTEL
    );
    private static final Money ONE_RUB = Money.of(1, ProtoCurrencyUnit.RUB);

    private final UserOrderCounterService userOrderCounterService;

    @Override
    @TransactionMandatory
    public CodeApplicationResult calculateDiscountForEstimation(PromoCodeDiscountCalculationCtx ctx) {
        String requestedPromocode = ctx.getUserRequestedCode();
        if (ctx.getServiceDescriptions().size() != 1) {
            return CodeApplicationResult.notApplicable(requestedPromocode);
        }

        ServiceDescription sd = ctx.getServiceDescriptions().get(0);
        if (!HOTEL_SERVICE_TYPES.contains(sd.getServiceType())) {
            return CodeApplicationResult.notApplicable(requestedPromocode);
        }

        DiscountApplicationConfig discountApplicationConfig = ctx.getPromoCode().getPromoAction()
                .getDiscountApplicationConfig();

        //TODO (mbobrov): implement discount application logic depending on the config

        HotelItinerary hotelItinerary = (HotelItinerary) sd.getPayload();

        if (!isApplicableForHotel(sd.getServiceType(), hotelItinerary.getOrderDetails().getOriginalId(),
                discountApplicationConfig)) {
            return CodeApplicationResult.notApplicable(requestedPromocode);
        }

        if (!discountApplicationConfig.isAddsUpWithOtherActions() && ctx.getAllPromoCodes().size() > 1) {
            return CodeApplicationResult.notApplicable(requestedPromocode);
        }

        if (!isApplicableForUserType(ctx.isUserStaff(), ctx.isUserPlus(), discountApplicationConfig)) {
            return CodeApplicationResult.notApplicable(requestedPromocode);
        }

        if (!applicableByMaxOrders(discountApplicationConfig, ctx.getPassportId(),
                mapServiceTypeToDisplayOrderType(sd.getServiceType()))) {
            return CodeApplicationResult.notApplicable(requestedPromocode);
        }

        if (discountApplicationConfig.getMinTotalCost() == null ||
                discountApplicationConfig.getMinTotalCost().isLessThanOrEqualTo(sd.getOriginalCost())) {
            Money discount = calculateDiscount(ctx.getPromoCode(), sd.getOriginalCost());
            if (discount == null) {
                return CodeApplicationResult.notApplicable(requestedPromocode);
            }
            BigDecimal remainingBudget = ctx.getPromoCode().getPromoAction().getRemainingBudget();
            if (remainingBudget == null || remainingBudget.subtract(discount.getNumberStripped()).compareTo(BigDecimal.ZERO) >= 0) {
                return CodeApplicationResult.success(requestedPromocode, discount);
            } else {
                return CodeApplicationResult.emptyBudget(requestedPromocode);
            }
        } else {
            return CodeApplicationResult.notApplicable(requestedPromocode);
        }
    }

    /**
     * @return promocode discount <code>null</code> if the code is not applicable.
     */
    private Money calculateDiscount(PromoCode code, Money originalCost) {
        BigDecimal nominal = code.getNominal();
        switch (code.getNominalType()) {
            case NT_VALUE:
                Money nominalDiscount = Money.of(nominal, ProtoCurrencyUnit.RUB);
                if (nominalDiscount.isLessThan(originalCost)) {
                    return nominalDiscount;
                } else {
                    // total price is 1 RUB
                    return originalCost.subtract(ONE_RUB);
                }
            case NT_PERCENT:
                Money percentDiscount = originalCost.divide(100).multiply(nominal);
                Money maxPossibleDiscount =
                        code.getPromoAction().getDiscountApplicationConfig().getMaxNominalDiscount();

                Money discount;
                if (maxPossibleDiscount != null && maxPossibleDiscount.isLessThan(percentDiscount)) {
                    discount = maxPossibleDiscount;
                } else {
                    discount = percentDiscount;
                }

                if (discount.isEqualTo(originalCost)) {
                    // total price is 1 RUB
                    return discount.subtract(ONE_RUB);
                } else {
                    return discount;
                }
            default:
                return null;
        }
    }

    /**
     * Checks if promo code can't be applied because it's applicable only for certain number of first orders.
     * Typical usage: if promo action is allowed only for the first order for new users.
     */
    @VisibleForTesting
    boolean applicableByMaxOrders(DiscountApplicationConfig discountApplicationConfig, Long passportId,
                                  EDisplayOrderType orderType) {
        if (discountApplicationConfig.limitedForFirstOrderOnlyFor(orderType)) {
            if (passportId == null) {
                return false;
            }
            return !userOrderCounterService.userHasOrdersConfirmed(passportId, orderType);
        }
        return true;
    }

    private boolean isApplicableForUserType(boolean userIsStaff, boolean userIsPlus,
                                            DiscountApplicationConfig discountApplicationConfig) {
        if (discountApplicationConfig.getUserTypeRestriction() != null) {
            boolean isApplicable = false;
            switch (discountApplicationConfig.getUserTypeRestriction()) {
                case PLUS_ONLY:
                    if (userIsPlus) {
                        isApplicable = true;
                    }
                    break;
                case STAFF_ONLY:
                    if (userIsStaff) {
                        isApplicable = true;
                    }
                    break;
                default:
                    throw new IllegalStateException(
                            String.format("Unknown user type restriction %s",
                                    discountApplicationConfig.getUserTypeRestriction())
                    );
            }
            return isApplicable;
        } else {
            return true;
        }
    }

    private boolean isApplicableForHotel(EServiceType partnerType, String originalId,
                                         DiscountApplicationConfig discountApplicationConfig) {
        if (!discountApplicationConfig.hasHotelRestrictions()) {
            return true;
        }
        for (HotelRestriction hotelRestriction : discountApplicationConfig.getHotelRestrictions()) {
            if (hotelRestriction.getPartner() == partnerType && hotelRestriction.getOriginalId().equals(originalId)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public FiscalItemApplicationResult calculateDiscountForApplication(PromoCodeDiscountApplicationCtx ctx) {
        if (ctx.getOrderItems().size() != 1) {
            return FiscalItemApplicationResult.notApplicable();
        }

        OrderItem oi = ctx.getOrderItems().get(0);
        if (!HOTEL_SERVICE_TYPES.contains(oi.getPublicType())) {
            return FiscalItemApplicationResult.notApplicable();
        }

        DiscountApplicationConfig discountApplicationConfig = ctx.getPromoCode().getPromoAction()
                .getDiscountApplicationConfig();

        //TODO (mbobrov): implement discount application logic depending on the config

        HotelItinerary hotelItinerary = (HotelItinerary) oi.getPayload();

        if (!isApplicableForHotel(oi.getPublicType(), hotelItinerary.getOrderDetails().getOriginalId(),
                discountApplicationConfig)) {
            return FiscalItemApplicationResult.notApplicable();
        }

        if (!discountApplicationConfig.isAddsUpWithOtherActions() && ctx.getAllPromoCodes().size() > 1) {
            return FiscalItemApplicationResult.notApplicable();
        }

        if (!isApplicableForUserType(ctx.isUserStaff(), ctx.isUserPlus(), discountApplicationConfig)) {
            return FiscalItemApplicationResult.notApplicable();
        }

        if (!applicableByMaxOrders(discountApplicationConfig, ctx.getPassportId(),
                mapServiceTypeToDisplayOrderType(oi.getPublicType()))) {
            return FiscalItemApplicationResult.notApplicable();
        }

        if (discountApplicationConfig.getMinTotalCost() == null ||
                discountApplicationConfig.getMinTotalCost().isLessThanOrEqualTo(oi.originalCostAfterReservation())) {
            Money totalOrderCost = oi.totalCostAfterReservationExceptPromo();
            Money discount = calculateDiscount(ctx.getPromoCode(), totalOrderCost);
            if (discount == null) {
                return FiscalItemApplicationResult.notApplicable();
            }
            BigDecimal remainingBudget = ctx.getPromoCode().getPromoAction().getRemainingBudget();
            if (remainingBudget == null || remainingBudget.subtract(discount.getNumberStripped()).compareTo(BigDecimal.ZERO) >= 0) {
                Map<FiscalItem, Money> discountMap = new HashMap<>(oi.getFiscalItems().size());
                for (var fi : oi.getFiscalItems()) {
                    Money itemShare =
                            fi.getMoneyAmountExceptPromo().divide(totalOrderCost.getNumber());
                    Money itemDiscount = MoneyUtils.roundToDecimal(discount.multiply(itemShare.getNumber()),
                            RoundingMode.HALF_UP);
                    discountMap.put(fi, itemDiscount);
                }
                Money discountsSum = discountMap.values().stream().reduce(Money.zero(totalOrderCost.getCurrency()),
                        Money::add);
                Money roundedDiscount = MoneyUtils.roundToDecimal(discount, RoundingMode.HALF_UP);
                Preconditions.checkState(discountsSum.isEqualTo(roundedDiscount),
                        "Sum of fiscal item discounts (%s) is not equal to (rounded) target discount (%s)",
                        discountsSum, roundedDiscount);
                return FiscalItemApplicationResult.success(discountMap);
            } else {
                return FiscalItemApplicationResult.emptyBudget();
            }
        } else {
            return FiscalItemApplicationResult.notApplicable();
        }
    }
}
