package ru.yandex.travel.orders.services.promo;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.commons.proto.EPromoCodeApplicationResultType;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.promo.PromoCode;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivation;
import ru.yandex.travel.orders.entities.promo.PromoCodeApplication;
import ru.yandex.travel.orders.entities.promo.PromoCodeBehaviourOverride;
import ru.yandex.travel.orders.repository.promo.PromoCodeActivationRepository;
import ru.yandex.travel.orders.repository.promo.PromoCodeRepository;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Component
@RequiredArgsConstructor
@Slf4j
public class PromoCodeApplicationService {

    private final PromoCodeRepository promoCodeRepository;

    private final PromoCodeActivationRepository promoCodeActivationRepository;

    private final PromoCodeDiscountCalculator discountCalculator;

    private final AuthorizationService authorizationService;

    private final Environment environment;

    private final PromoCodeChecker checker;

    @TransactionMandatory
    public PromoCodeApplicationResult calculateResult(UserCredentials userCredentials,
                                                      PromoCodeCalculationRequest request) {
        // find relevant promo codes
        // check that they're not used by current user
        // filter non valid promo-codes
        // for each if can be combined - apply to the resulting sum

        Preconditions.checkArgument(!Strings.isNullOrEmpty(userCredentials.getPassportId()), "Passport id must be " +
                "present");

        Money totalCost = Money.zero(ProtoCurrencyUnit.RUB);

        for (ServiceDescription sd : request.getServiceDescriptions()) {
            totalCost = totalCost.add(sd.getOriginalCost());
        }

        List<String> codeCandidates = new ArrayList<>(request.getPromoCodes());
        Map<String, CodeApplicationResult> codeApplicationResultMap = new HashMap<>();

        List<PromoCodeWithCodeTuple> foundPromoCodes = new ArrayList<>();

        for (String userRequestedCode : codeCandidates) {
            PromoCode promoCode = promoCodeRepository.findByCodeEquals(PromoCodeUnifier.unifyCode(userRequestedCode));

            if (promoCode == null) {
                codeApplicationResultMap.put(userRequestedCode, CodeApplicationResult.notFound(userRequestedCode));
                continue;
            }

            ApplicationResultType promoCodeCheckResult = checker.checkPromoCode(promoCode,
                    promoCodeActivationRepository.lookupActivationForPromoCodeAndPassportId(
                            userCredentials.getPassportId(),
                            promoCode.getId()
                    ));
            if (promoCodeCheckResult != ApplicationResultType.SUCCESS) {
                codeApplicationResultMap.put(userRequestedCode,
                        new CodeApplicationResult(userRequestedCode, promoCodeCheckResult, null));
            } else {
                foundPromoCodes.add(new PromoCodeWithCodeTuple(userRequestedCode, promoCode));
            }
        }

        Money discount = Money.zero(ProtoCurrencyUnit.RUB);

        List<PromoCode> allFoundPromoCodes =
                foundPromoCodes.stream().map(PromoCodeWithCodeTuple::getPromoCode).collect(Collectors.toUnmodifiableList());

        for (PromoCodeWithCodeTuple tuple : foundPromoCodes) {
            PromoCode promoCode = tuple.promoCode;

            PromoCodeDiscountCalculationCtx ctx = new PromoCodeDiscountCalculationCtx();
            ctx.setUserRequestedCode(tuple.code);
            ctx.setUserPlus(userCredentials.isUserIsPlus());
            ctx.setUserStaff(userCredentials.isUserIsStaff());
            ctx.setServiceDescriptions(request.getServiceDescriptions());
            ctx.setPromoCode(promoCode);
            ctx.setAllPromoCodes(allFoundPromoCodes);

            if (!Strings.isNullOrEmpty(userCredentials.getPassportId())) {
                ctx.setPassportId(Long.parseLong(userCredentials.getPassportId()));
            }

            CodeApplicationResult promoApplicationResult = discountCalculator.calculateDiscountForEstimation(ctx);
            if (promoApplicationResult.getType() == ApplicationResultType.SUCCESS) {
                discount = discount.add(promoApplicationResult.getDiscountAmount());
            }
            codeApplicationResultMap.put(promoApplicationResult.getCode(), promoApplicationResult);
        }

        Preconditions.checkState(codeCandidates.size() == codeApplicationResultMap.size(),
                "Not all promo code application results known");

        List<CodeApplicationResult> codeApplicationResults = new ArrayList<>();
        for (String code : codeCandidates) {
            codeApplicationResults.add(codeApplicationResultMap.get(code));
        }
        PromoCodeApplicationResult result = new PromoCodeApplicationResult();
        result.setApplicationResults(codeApplicationResults);
        result.setOriginalAmount(totalCost);
        result.setDiscountedAmount(totalCost.subtract(discount));
        result.setDiscountAmount(discount);
        return result;
    }

    /**
     * The result should be checked in {@code order.promoCodeApplications.applicationResultType}
     *
     * @param toCountApplication whether the application should be counted (in promo action budget, in promo
     *                           activation).
     * @implNote ATM not quite clear what's the conceptual difference between this method with toCountApplication=false
     * and {@link #calculateResult(UserCredentials, PromoCodeCalculationRequest)}, but it seems too challenging now
     * to refactor the code here and in related files.
     */
    @TransactionMandatory
    public void usePromoCodeActivationsAndApplyDiscounts(Order order, boolean toCountApplication) {

        List<PromoCode> allPromoCodes = order.getPromoCodeApplications()
                .stream().map(ca -> ca.getPromoCodeActivation().getPromoCode())
                .collect(Collectors.toUnmodifiableList());

        AuthorizedUser authorizedUser = authorizationService.getRequiredOrderOwner(order.getId());

        for (PromoCodeApplication application : order.getPromoCodeApplications()) {
            PromoCode promoCode = application.getPromoCodeActivation().getPromoCode();

            // OVERRIDES BEHAVIOUR FOR TEST PURPOSES (restricted for prod env)
            if (!environment.acceptsProfiles("prod")) {
                if (promoCode.getBehaviourOverride() == PromoCodeBehaviourOverride.RESTRICT_APPLICATION) {
                    application.setApplicationResultType(EPromoCodeApplicationResultType.ART_NOT_APPLICABLE);
                    continue;
                }
                if (promoCode.getBehaviourOverride() == PromoCodeBehaviourOverride.RESTRICT_ALREADY_APPLIED) {
                    application.setApplicationResultType(EPromoCodeApplicationResultType.ART_ALREADY_APPLIED);
                    continue;
                }
            }

            ApplicationResultType applicationResultType =
                    checker.checkPromoCode(promoCode, application.getPromoCodeActivation());
            if (applicationResultType != ApplicationResultType.SUCCESS) {
                application.setApplicationResultType(
                        applicationResultType.getProtoValue()
                );
                continue;
            }


            PromoCodeDiscountApplicationCtx ctx = new PromoCodeDiscountApplicationCtx();
            ctx.setAllPromoCodes(allPromoCodes);
            ctx.setUserPlus(authorizedUser.getUserIsPlus());
            ctx.setUserStaff(authorizedUser.getUserIsStaff());
            ctx.setPromoCode(application.getPromoCodeActivation().getPromoCode());
            ctx.setOrderItems(order.getOrderItems());
            if (!Strings.isNullOrEmpty(authorizedUser.getPassportId())) {
                ctx.setPassportId(Long.parseLong(authorizedUser.getPassportId()));
            }
            FiscalItemApplicationResult fiscalItemApplicationResult =
                    discountCalculator.calculateDiscountForApplication(ctx);
            log.info("calculated promo code application result for order {}: {}",
                    order.getId(), fiscalItemApplicationResult);
            if (fiscalItemApplicationResult.getResultType() == ApplicationResultType.SUCCESS) {
                application.setDiscountIfNull(fiscalItemApplicationResult.getDiscountMap());
                application.setApplicationResultType(EPromoCodeApplicationResultType.ART_SUCCESS);
                if (toCountApplication) {
                    promoCode.getPromoAction().subtractFromBudget(application);
                    application.getPromoCodeActivation().incrementTimesUsed();
                    application.setAppliedAt(Instant.now());
                }
            } else if (fiscalItemApplicationResult.getResultType() == ApplicationResultType.NOT_APPLICABLE) {
                application.setApplicationResultType(EPromoCodeApplicationResultType.ART_NOT_APPLICABLE);
            } else if (fiscalItemApplicationResult.getResultType() == ApplicationResultType.EMPTY_BUDGET) {
                application.setApplicationResultType(EPromoCodeApplicationResultType.ART_EMPTY_BUDGET);
            }
        }
    }

    @TransactionMandatory
    public void freePromoCodeActivations(Order order) {
        for (PromoCodeApplication application : order.getPromoCodeApplications()) {
            if (application.hasBeenActivated()) {
                log.info("Releasing promo code activation {} for order {}, {}",
                        application.getId(),
                        order.getId(),
                        order.getPrettyId());
                PromoCodeActivation activation = application.getPromoCodeActivation();
                activation.decrementTimesUsed();

                activation.getPromoCode()
                        .getPromoAction()
                        .restoreBudgetSpending(application);
            } else {
                log.info("Not releasing promo code activation {} for order {}, {}. " +
                                "Looks like it had never been activated.",
                        application.getId(),
                        order.getId(),
                        order.getPrettyId());
            }
        }
    }

    // class used to hold code requested by user and a corresponding entity found in our database
    @Value
    private static class PromoCodeWithCodeTuple {
        String code; // requested
        PromoCode promoCode;
    }
}
