package ru.yandex.travel.orders.services.promo;

import java.time.Clock;
import java.time.Instant;

import javax.annotation.Nullable;

import com.google.common.annotations.VisibleForTesting;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ErrorException;
import ru.yandex.travel.orders.entities.promo.PromoCode;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivation;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivationsStrategy;
import ru.yandex.travel.orders.entities.promo.PromoCodeHelpers;

@Service
@Slf4j
@RequiredArgsConstructor
public class PromoCodeChecker {
    private final Clock clock;

    /**
     * see {@link #checkPromoCode(PromoCode, PromoCodeActivation)}
     */
    public ApplicationResultType checkPromoCode(PromoCode promoCode) {
        return checkPromoCode(promoCode, null);
    }

    /**
     * Checks if promo code can be applied (not blacklisted, etc.)
     *
     * @param activation if is not provided, the checker checks only information in promoCode
     */
    public ApplicationResultType checkPromoCode(PromoCode promoCode,
                                                @Nullable PromoCodeActivation activation) {
        if (promoCode == null) {
            return ApplicationResultType.NOT_FOUND;
        }
        if (promoCode.isBlacklisted()) {
            return ApplicationResultType.NOT_APPLICABLE;
        }
        Instant now = Instant.now(clock);
        if (notStarted(promoCode, now)) {
            // NOT_FOUND had been used before for the status. But seems like only
            // the mapped proto enum value matters outside of the app.
            return ApplicationResultType.NOT_STARTED;
        }
        if (expired(promoCode, now)) {
            return ApplicationResultType.EXPIRED;
        }
        if (alreadyApplied(promoCode, activation)) {
            return ApplicationResultType.ALREADY_APPLIED;
        }
        return ApplicationResultType.SUCCESS;
    }

    private boolean notStarted(PromoCode promoCode, Instant now) {
        Instant validFrom = PromoCodeHelpers.getPromoCodeValidFrom(promoCode);
        if (validFrom != null) {
            boolean started = validFrom.isBefore(now);
            return !started;
        }
        return false;
    }

    private boolean expired(PromoCode promoCode, Instant now) {
        Instant validTill = PromoCodeHelpers.getPromoCodeValidTill(promoCode);
        if (validTill != null) {
            return validTill.isBefore(now);
        }
        return false;
    }

    /**
     * The method exploits the following {@link PromoCode}'s fields: allowedUsageCount,
     * activationStrategy, allowedActivationsTotal, allowedActivationsCount.
     * <p>
     * See corresponding fields or the method's implementation for details.
     */
    @VisibleForTesting
    boolean alreadyApplied(PromoCode promoCode, @Nullable PromoCodeActivation activation) {
        if (promoCode.getActivationsStrategy() == PromoCodeActivationsStrategy.LIMITED_ACTIVATIONS) {
            int totalRemains = promoCode.getAllowedActivationsTotal() - promoCode.getAllowedActivationsCount();
            // if there's no activation linked, we suppose that a new activation should be created and therefore reject
            if (activation == null &&
                    totalRemains == 0) {
                return true;
            }
            if (totalRemains < 0) {
                log.error("The promo code had been exploited more than allowed. Please check the DB! id={}, code={}",
                        promoCode.getId(),
                        promoCode.getCode());
                return true;
            }
        }

        // if the activation exists, we should check if the user can reuse the promocode
        if (activation != null) {
            return activation.getTimesUsed() >= promoCode.getAllowedUsageCount();
        } else if (promoCode.getAllowedUsageCount() <= 0) {
            log.error("Promo code's allowed usage count has incorrect value. Please check the DB! id={}, code={}",
                    promoCode.getId(),
                    promoCode.getCode());
            return true;
        }
        return false;
    }

    /**
     * Checks as {@link #checkPromoCode(PromoCode)}, but throws an {@link ErrorException} in case of an error
     */
    public void ensurePromoCodeCanBeActivated(PromoCode promoCode) throws ErrorException {
        ApplicationResultType checkResult = checkPromoCode(promoCode);
        if (checkResult != ApplicationResultType.SUCCESS) {
            throw Error.with(
                    EErrorCode.EC_FAILED_PRECONDITION,
                    checkResult.getErrorMessage(promoCode.getCode())
            ).toEx();
        }
    }
}
