package ru.yandex.travel.orders.services.promo;

import java.math.BigDecimal;
import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Optional;
import java.util.UUID;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.commons.proto.EPromoCodeNominalType;
import ru.yandex.travel.orders.entities.promo.BasePromoCodeGenerationConfig;
import ru.yandex.travel.orders.entities.promo.PromoAction;
import ru.yandex.travel.orders.entities.promo.PromoCode;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivationsStrategy;
import ru.yandex.travel.orders.repository.promo.PromoActionRepository;
import ru.yandex.travel.orders.repository.promo.PromoCodeRepository;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Service
@RequiredArgsConstructor
public class PromoCodeGenerationService {
    public static final ZoneId MSK_TZ = ZoneId.of("Europe/Moscow");
    private final PromoCodeGenerationStrategyProvider generationStrategyProvider;
    private final PromoActionRepository promoActionRepository;
    private final PromoCodeRepository promoCodeRepository;
    private final Clock clock;

    @TransactionMandatory
    public PromoCode generatePromoCodeForAction(UUID actionId, LocalDate forDate) {
        Preconditions.checkArgument(actionId != null, "ActionID should be present");
        PromoAction action = promoActionRepository.getOne(actionId);
        return generatePromoCodeForAction(action, forDate);
    }

    @TransactionMandatory
    public PromoCode generatePromoCodeForAction(String actionName, LocalDate forDate) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(actionName), "Action name should be present");
        Optional<PromoAction> maybeAction = promoActionRepository.findByName(actionName);
        return maybeAction.map(promoAction -> generatePromoCodeForAction(promoAction, forDate)).orElse(null);
    }

    @TransactionMandatory
    public PromoCode generatePromoCodeForAction(PromoAction promoAction, LocalDate forDate) {
        Preconditions.checkArgument(promoAction != null, "Promo action should be present");
        Preconditions.checkArgument(promoAction.getPromoCodeGenerationConfig() != null,
                "Promo action should have generation config set");
        BasePromoCodeGenerationConfig config = promoAction.getPromoCodeGenerationConfig();

        return generatePromoCodeForAction(promoAction, BigDecimal.valueOf(config.getNominal()), config.getNominalType(), forDate);
    }

    @TransactionMandatory
    public PromoCode generatePromoCodeForAction(PromoAction promoAction,
                                                BigDecimal nominal,
                                                EPromoCodeNominalType nominalType,
                                                LocalDate forDate) {
        Preconditions.checkArgument(promoAction != null, "Promo action should be present");
        Preconditions.checkArgument(promoAction.getPromoCodeGenerationConfig() != null,
                "Promo action should have generation config set");
        BasePromoCodeGenerationConfig config = promoAction.getPromoCodeGenerationConfig();
        if (promoAction.getValidFrom() != null && promoAction.getValidFrom().isAfter(Instant.now(clock))) {
            // Do not generate promo code if the action hasn't started yet
            return null;
        }
        PromoCode promoCode = new PromoCode();
        promoCode.setId(UUID.randomUUID());
        promoCode.setPromoAction(promoAction);

        PromoCodeGenerationStrategy generationStrategy =
                generationStrategyProvider.getStrategyForGenerationType(promoAction.getPromoCodeGenerationType());
        promoCode.setCode(generationStrategy.generatePromoCodeForAction(promoAction, forDate));

        promoCode.setNominal(nominal == null ? BigDecimal.valueOf(config.getNominal()) : nominal);
        promoCode.setNominalType(nominalType == null ? config.getNominalType() : nominalType);
        promoCode.setAllowedUsageCount(config.getMaxUsagePerUser());
        promoCode.setAllowedActivationsCount(0);

        if (config.getMaxActivations() > 0) {
            promoCode.setActivationsStrategy(PromoCodeActivationsStrategy.LIMITED_ACTIVATIONS);
            promoCode.setAllowedActivationsTotal(config.getMaxActivations());
        } else {
            promoCode.setActivationsStrategy(PromoCodeActivationsStrategy.UNLIMITED_ACTIVATIONS);
        }

        Instant now = atStartDayInMoscow(LocalDate.now(clock));
        promoCode.setValidFrom(promoAction.getValidFrom() != null && promoAction.getValidFrom().isAfter(now)
                ? promoAction.getValidFrom()
                : now);

        switch (config.getValidTillGenerationType()) {
            case FIXED_DATE:
                Instant fixedDateInstant = atEndOfDayInMoscow(config.getFixedDate());
                promoCode.setValidTill(promoAction.getValidTill().isAfter(fixedDateInstant) ? promoAction.getValidTill() : fixedDateInstant);
                break;
            case FIXED_DURATION:
                LocalDate validFromDate = promoCode.getValidFrom().atZone(MSK_TZ).toLocalDate();
                Instant calculatedValidTill = atEndOfDayInMoscow(validFromDate.plusDays(config.getFixedDaysDuration()));
                promoCode.setValidTill(calculatedValidTill);
                break;
            default:
                break;
        }
        promoCode = promoCodeRepository.save(promoCode);
        return promoCode;
    }

    private static Instant atStartDayInMoscow(LocalDate date) {
        if (date == null) {
            return null;
        }
        return date.atStartOfDay(MSK_TZ).toInstant();
    }

    @VisibleForTesting
    static Instant atEndOfDayInMoscow(LocalDate date) {
        if (date == null) {
            return null;
        }
        return date.atTime(23, 59, 59).atZone(MSK_TZ).toInstant();
    }
}
