package ru.yandex.travel.orders.services.promo;

import java.math.BigInteger;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.promo.PromoAction;
import ru.yandex.travel.orders.entities.promo.SimplePromoCodeGenerationConfig;
import ru.yandex.travel.orders.repository.promo.PromoActionRepository;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Service
@RequiredArgsConstructor
public class SimplePromoCodeGenerationStrategy implements PromoCodeGenerationStrategy {
    private final PromoActionRepository promoActionRepository;

    @Override
    @TransactionMandatory
    public String generatePromoCodeForAction(UUID promoActionId, LocalDate date) {
        PromoAction action = promoActionRepository.getOne(promoActionId);
        return generatePromoCodeForAction(action, date);
    }

    @Override
    @TransactionMandatory
    public String generatePromoCodeForAction(PromoAction promoAction, LocalDate date) {
        var config = (SimplePromoCodeGenerationConfig) promoAction.getPromoCodeGenerationConfig();
        Long sequenceValue = promoActionRepository.getNextSimpleGenerationStrategySequenceValue();

        return PromoCodeUnifier.unifyCode(constructPromoCode(config.getPrefix(), date, sequenceValue, config.getSuffix()));
    }

    private static String constructPromoCode(String prefix, LocalDate date, Long prngValue, String suffix) {
        StringBuilder promoCode = new StringBuilder();
        promoCode.append(prefix).append("-");

        BigInteger datePart = getDatePart(date);
        BigInteger prngPart = getPrngPart(prngValue);
        String dictPart = getDictPart(datePart, prngPart);
        String checksum = getChecksum(datePart, prngPart);
        promoCode.append(dictPart).append("-").append(checksum);

        promoCode.append("-").append(suffix);

        return promoCode.toString();
    }

    private static String getDictPart(BigInteger datePart, BigInteger prngPart) {
        return dict(zip(datePart, prngPart));
    }

    private static String getChecksum(BigInteger datePart, BigInteger prngPart) {
        byte[] bytes = ArrayUtils.addAll(datePart.toByteArray(), prngPart.toByteArray());
        return checksum(bytes).toUpperCase();
    }

    private static BigInteger getDatePart(LocalDate date) {
        var duration = DATE_BASE.until(date, ChronoUnit.DAYS);
        return BigInteger.valueOf(duration);
    }

    private static BigInteger getPrngPart(long id) {
        return PRNG_BASE.modPow(BigInteger.valueOf(id), PRNG_MODULUS);
    }

    private static BigInteger zip(BigInteger first, BigInteger second) {
        if (first == null) {
            return second;
        } else if (second == null) {
            return first;
        } else {
            int maxLength = Integer.max(first.bitLength(), second.bitLength());
            BigInteger result = BigInteger.ZERO;
            var k = 0;
            for (var i = 0; i <= maxLength; i++) {
                if (first.testBit(i)) {
                    result = result.setBit(k);
                }
                k = k + 1;
                if (second.testBit(i)) {
                    result = result.setBit(k);
                }
                k = k + 1;
            }
            return result;
        }
    }

    private static String dict(BigInteger word) {
        if (word == null) {
            return "";
        }
        var absWord = word.abs();
        StringBuilder result = new StringBuilder();
        var divRem = absWord.divideAndRemainder(DICT_BASE);
        while (!divRem[0].equals(BigInteger.ZERO)) {
            result.append(DICT.get(divRem[1].intValueExact()));
            divRem = divRem[0].divideAndRemainder(DICT_BASE);
        }
        result.append(DICT.get(divRem[1].intValueExact()));
        return result.reverse().toString();
    }

    private static final BigInteger PRNG_BASE = BigInteger.valueOf(1000211L);
    private static final BigInteger PRNG_MODULUS = BigInteger.valueOf(1000213L);
    private static final LocalDate DATE_BASE = LocalDate.of(2020, 2, 1);
    private static final BigInteger DICT_BASE = BigInteger.valueOf(30);
    private static final Map<Integer, Character> DICT = Map.ofEntries(
            Map.entry(0, '0'), Map.entry(1, '1'), Map.entry(2, '2'), Map.entry(3, '3'),
            Map.entry(4, '4'), Map.entry(5, '5'), Map.entry(6, '6'), Map.entry(7, '7'),
            Map.entry(8, '8'), Map.entry(9, '9'), Map.entry(10, 'B'), Map.entry(11, 'C'),
            Map.entry(12, 'D'), Map.entry(13, 'F'), Map.entry(14, 'G'), Map.entry(15, 'H'),
            Map.entry(16, 'J'), Map.entry(17, 'K'), Map.entry(18, 'L'), Map.entry(19, 'M'),
            Map.entry(20, 'N'), Map.entry(21, 'P'), Map.entry(22, 'Q'), Map.entry(23, 'R'),
            Map.entry(24, 'S'), Map.entry(25, 'T'), Map.entry(26, 'V'), Map.entry(27, 'W'),
            Map.entry(28, 'X'), Map.entry(29, 'Z')
    );

    private static String checksum(byte[] bytes) {
        int newCrc;
        int i;
        newCrc = 0xFF;
        for(i = 0; i < bytes.length; i++) {
            newCrc = crc16_update(newCrc, bytes[i]);
        }
        return Integer.toHexString(Math.abs(newCrc));
    }

    private static int crc16_update(int crc, int a)
    {
        int i;
        crc ^= a;

        for (i = 0; i < 8; ++i) {
            if ((crc & 1) == 1){
                crc = ((crc >> 1) ^ 0xA1);
            }
            else{
                crc = (crc >> 1);
            }
        }
        return crc;
    }
}
