package ru.yandex.travel.orders.services.promo;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.UserOrderCounter;
import ru.yandex.travel.orders.repository.AuthorizedUserRepository;
import ru.yandex.travel.orders.repository.UserOrderCounterRepository;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Slf4j
@Service
@RequiredArgsConstructor
public class UserOrderCounterService {
    private final UserOrderCounterRepository userOrderCounterRepository;
    private final AuthorizedUserRepository authorizedUserRepository;

    public int getNumberOfHotelOrdersConfirmedByUser(long passportId) {
        return getNumberOfOrdersConfirmedByUser(passportId, EDisplayOrderType.DT_HOTEL);
    }

    /**
     * Returns number of orders made by a user for the provided order type.
     *
     * Probably {@link #userHasOrdersConfirmed(long, EDisplayOrderType)} would be a better fit.
     */
    public int getNumberOfOrdersConfirmedByUser(long passportId, EDisplayOrderType type) {
        return userOrderCounterRepository.findById(passportId).map(counter -> counter.getConfirmedOrders(type))
                .orElse(0);
    }

    public boolean userHasOrdersConfirmed(long passportId, EDisplayOrderType type) {
        return this.getNumberOfOrdersConfirmedByUser(passportId, type) > 0;
    }

    public List<EDisplayOrderType> getUserExistingOrderTypes(String passportId) {
        if (Strings.isNullOrEmpty(passportId)) {
            return Collections.emptyList();
        }
        long passportIdParsed;
        try {
            passportIdParsed = Long.parseLong(passportId);
        } catch (NumberFormatException e) {
            return Collections.emptyList();
        }
        return Arrays.stream(EDisplayOrderType.values())
                .filter(ot -> ot != EDisplayOrderType.UNRECOGNIZED &&
                        ot != EDisplayOrderType.DT_UNKNOWN &&
                        userHasOrdersConfirmed(passportIdParsed, ot))
                .collect(Collectors.toList());
    }

    @TransactionMandatory
    public void onOrderConfirmed(UUID orderId, EDisplayOrderType type) {
        getOrInitOrderCounter(orderId).ifPresent(counter -> counter.orderConfirmed(type));
    }

    @TransactionMandatory
    public void onOrderConfirmed(Order order) {
        onOrderConfirmed(order.getId(), order.getDisplayType());
    }

    @TransactionMandatory
    public void onOrderRefunded(UUID orderId, EDisplayOrderType type) {
        getOrInitOrderCounter(orderId).ifPresent(counter -> counter.orderRefunded(type));
    }

    @TransactionMandatory
    public void onOrderRefunded(Order order) {
        onOrderRefunded(order.getId(), order.getDisplayType());
    }

    private Optional<UserOrderCounter> getOrInitOrderCounter(UUID orderId) {
        List<AuthorizedUser> ownersList = authorizedUserRepository.findByIdOrderIdAndRole(
                orderId, AuthorizedUser.OrderUserRole.OWNER
        );

        Preconditions.checkState(ownersList.size() == 1, "Order %s must have exactly one owner", orderId);

        AuthorizedUser owner = ownersList.get(0);
        if (Strings.isNullOrEmpty(owner.getPassportId())) {
            return Optional.empty();
        }
        long passportId = Long.parseLong(owner.getPassportId());
        Optional<UserOrderCounter> maybeOrderCounter =
                userOrderCounterRepository.findById(passportId);
        if (maybeOrderCounter.isPresent()) {
            return maybeOrderCounter;
        } else {
            UserOrderCounter result = UserOrderCounter.initForUser(passportId);
            return Optional.of(userOrderCounterRepository.save(result));
        }
    }
}
