package ru.yandex.travel.orders.services.promo.aeroflotplus;

import java.time.Clock;
import java.time.Instant;
import java.util.List;
import java.util.UUID;

import javax.annotation.PostConstruct;

import com.google.common.base.Preconditions;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.promo.AeroflotPlusPromoInfo;
import ru.yandex.avia.booking.promo.AeroflotPlusPromoInfo.PlusCode;
import ru.yandex.avia.booking.promo.AviaPromoCampaignsInfo;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.notifications.MailTarget;
import ru.yandex.travel.orders.entities.promo.aeroflotplus.AeroflotPlusPromoCode;
import ru.yandex.travel.orders.repository.promo.aeroflotplus.AeroflotPlusPromoCodeRepository;
import ru.yandex.travel.orders.services.email.AsyncEmailOperation;
import ru.yandex.travel.orders.services.email.SendEmailParams;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotPlusPromoService {
    private final AeroflotPlusPromoProperties properties;
    private final AeroflotPlusPromoCodeRepository promoCodeRepository;
    private final SingleOperationService singleOperationService;
    private final Clock clock;

    @PostConstruct
    public void init() {
        Preconditions.checkArgument(!properties.getStartsAt().isAfter(properties.getEndsAt()),
                "The promo start date can't be after the promo end date");
    }

    @TransactionMandatory
    public void registerConfirmedOrder(AeroflotOrder order) {
        if (!isPromoEnabled()) {
            log.info("Aeroflot Plus Promo isn't enabled, skipping the order");
            return;
        }
        AeroflotPlusPromoInfo orderPromo = getPlusPromo(order.getAeroflotOrderItem());
        if (orderPromo == null || !orderPromo.isEnabled()) {
            log.info("This order is not eligible for Aeroflot Plus Promo");
            return;
        }
        log.info("Aeroflot Plus Promo is enabled preparing the email for {}", order.getEmail());
        assignPromoCodes(order, orderPromo);
        scheduleEmail(order, orderPromo);
    }

    private boolean isPromoEnabled() {
        Instant now = Instant.now(clock);
        return properties.getStartsAt().isBefore(now) && now.isBefore(properties.getEndsAt());
    }

    private AeroflotPlusPromoInfo getPlusPromo(AeroflotOrderItem service) {
        AviaPromoCampaignsInfo promoCampaignsInfo = service.getPayload().getPromoCampaignsInfo();
        if (promoCampaignsInfo == null) {
            return null;
        }
        return promoCampaignsInfo.getPlusPromo2021();
    }

    private void assignPromoCodes(AeroflotOrder order, AeroflotPlusPromoInfo promo) {
        List<PlusCode> promisedCodes = promo.getPlusCodes();
        log.info("{} plus promo codes will be assigned: {}", promisedCodes.size(), promisedCodes);
        // todo(tlg-13): most likely we'll have to set the number of code to 1 according to the latest promo update
        Preconditions.checkArgument(0 < promisedCodes.size() && promisedCodes.size() <= 10,
                "Unexpected amount of codes promised - %s", promisedCodes.size());
        List<AeroflotPlusPromoCode> codesToUse = promoCodeRepository.findFreeCodes(promisedCodes.size());
        for (int i = 0; i < promisedCodes.size(); i++) {
            PlusCode promisedCode = promisedCodes.get(i);
            AeroflotPlusPromoCode codeToUse = codesToUse.get(i);
            Preconditions.checkState(promisedCode.getPoints() == codeToUse.getPlusPoints(),
                    "Unexpected amount of points per code was promised; promised %s, actual %s",
                    promisedCode.getPoints(), codeToUse.getPlusPoints());
            Preconditions.checkState(promisedCode.getCode() == null, "Some promo code has already been assigned");
            codeToUse.setUsedAt(Instant.now(clock));
            codeToUse.setOrderId(order.getId());
            // usually we should update an item only from its workflow to prevent OptimisticLockingException-s
            // but the promo is short and we don't have too much time to improve every little detail
            promisedCode.setCode(codeToUse.getCode());
        }
    }

    private void scheduleEmail(AeroflotOrder order, AeroflotPlusPromoInfo promo) {
        // it's important to schedule the non-transactional email service call in a separate transaction
        // as the current transaction may be rolled back a few times because of optimistic locking exceptions
        // during commit due to concurrent promo code assignment attempts and semi-legal order item updates

        List<PlusCode> codes = promo.getPlusCodes();
        Preconditions.checkArgument(codes.size() == 1, "Unexpected amount of generated codes: %s", codes.size());
        String plusCode = codes.get(0).getCode();
        SendEmailParams emailData = SendEmailParams.builder()
                .campaignId(properties.getEmailCampaign())
                .targets(List.of(new MailTarget(order.getEmail())))
                .arguments(SendEmailParams.wrapArgs(new EmailTemplateArgs(plusCode)))
                .contextEntityId(order.getId())
                .build();
        UUID operationId = singleOperationService.runOperation(
                "AeroflotPlusPromoSendEmail_" + Instant.now(clock),
                AsyncEmailOperation.TYPE.getValue(),
                emailData
        );
        Preconditions.checkState(promo.getEmailTaskId() == null, "Some email task id has already been set");
        // usually we should update an item only from its workflow to prevent OptimisticLockingException-s
        // but the promo is short and we don't have too much time to improve every little detail
        promo.setEmailTaskId(operationId);
    }

    @RequiredArgsConstructor
    @Getter
    static class EmailTemplateArgs {
        private final String code;
    }
}
