package ru.yandex.travel.orders.services.promo.export;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.persistence.EntityManager;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;

import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.transactions.utils.YtTransactionsUtils;
import ru.yandex.inside.yt.kosher.tables.types.NativeProtobufEntryType;
import ru.yandex.travel.commons.metrics.MetricsUtils;
import ru.yandex.travel.orders.entities.UserOrderCounter;
import ru.yandex.travel.orders.repository.UserOrderCounterRepository;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.yt.util.YtHelper;
import ru.yandex.travel.yt.util.YtProtoUtils;

@Slf4j
public abstract class AbstractUserOrderCounterYtExporter<T extends Message> {
    private static final boolean PING_ANCESTOR_TRANSACTION = false;
    private static final Optional<Duration> transactionPingInterval = Optional.of(Duration.ofSeconds(10));
    private final String taskKey;
    private final Map<String, Yt> ytClusters;
    private final Map<String, Timer> clusterTimers = new HashMap<>();
    private final EntityManager entityManager;
    protected final UserOrderCounterYtExporterPropertiesBase properties;
    protected final UserOrderCounterRepository userOrderCounterRepository;
    private final T protoDefaultInstance;

    public AbstractUserOrderCounterYtExporter(UserOrderCounterYtExporterPropertiesBase properties,
                                              UserOrderCounterRepository userOrderCounterRepository,
                                              EntityManager entityManager,
                                              String taskKey,
                                              T protoDefaultInstance) {
        this.taskKey = taskKey;
        this.properties = properties;
        this.userOrderCounterRepository = userOrderCounterRepository;
        this.entityManager = entityManager;
        this.protoDefaultInstance = protoDefaultInstance;

        ytClusters = properties.getClusters().stream()
                .collect(Collectors.toUnmodifiableMap(Function.identity(),
                        cluster -> YtUtils.http(cluster, properties.getToken())));
        ytClusters.keySet().forEach(cluster -> clusterTimers.put(cluster,
                Timer.builder(String.format("%s.exportTime", taskKey))
                        .tag("ytCluster", cluster)
                        .serviceLevelObjectives(MetricsUtils.mediumDurationSla())
                        .publishPercentiles(MetricsUtils.higherPercentiles())
                        .publishPercentileHistogram().register(Metrics.globalRegistry))
        );
    }

    public String getTaskKey() {
        return taskKey;
    }

    protected abstract Iterator<T> mapUserOrderCountersToProto(List<UserOrderCounter> countersBatch);

    protected abstract List<UserOrderCounter> getCountersBatch(PageRequest pageRequest);

    @TransactionMandatory
    public void exportUserOrderCounterToYt(String taskId) {
        List<Exception> occurredExceptions = new ArrayList<>();
        for (Map.Entry<String, Yt> entry : ytClusters.entrySet()) {
            String cluster = entry.getKey();
            Yt yt = entry.getValue();
            long startTime = System.nanoTime();
            try {
                YtTransactionsUtils.withTransaction(
                        yt,
                        properties.getTransactionDuration(),
                        transactionPingInterval,
                        tx -> {
                            var txId = tx.getId();
                            log.info("Exporting user order counters using yt transaction [{}]", txId);
                            Preconditions.checkArgument(taskKey.equals(taskId));
                            this.internalExportUserOrderCounterToYt(txId, yt);
                            log.info("Done: Exporting user order counters using yt transaction [{}]", txId);
                            return null;
                        });
            } catch (Exception e) {
                occurredExceptions.add(e);
            }
            clusterTimers.get(cluster).record(System.nanoTime() - startTime, TimeUnit.NANOSECONDS);
        }
        if (!occurredExceptions.isEmpty()) {
            RuntimeException resultingException = new RuntimeException(String.format("%d exceptions occurred while " +
                    "exporting user order counters to yt clusters", occurredExceptions.size()));
            for (Exception exception : occurredExceptions) {
                resultingException.addSuppressed(exception);
            }
            throw resultingException;
        }
    }

    private void internalExportUserOrderCounterToYt(GUID txId, Yt yt) {
        var userOrderCounterType = new NativeProtobufEntryType<T>(protoDefaultInstance.newBuilderForType());
        YPath userOrderCountersTable = YtHelper.createTable(yt, txId, PING_ANCESTOR_TRANSACTION,
                        properties.getUserOrderCountersTablePath(),
                        YtProtoUtils.getTableSchemaForMessage(protoDefaultInstance))
                .append(true);

        int page = 0;
        while (true) {
            List<UserOrderCounter> countersBatch = getCountersBatch(
                    PageRequest.of(page,
                            properties.getDbBatchSize(),
                            Sort.by(Sort.Direction.ASC, "passportId")));
            if (countersBatch.isEmpty()) {
                break;
            }

            Iterator<T> countersRecords = mapUserOrderCountersToProto(countersBatch);
            yt.tables().write(Optional.of(txId), PING_ANCESTOR_TRANSACTION,
                    userOrderCountersTable, userOrderCounterType, countersRecords);
            page++;
            // removing counters from the entityManager cache
            countersBatch.forEach(entityManager::detach);
        }
    }
}
