package ru.yandex.travel.orders.services.promo.mir2020;

import java.time.Instant;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.common.promo.mir.MirEligibilityService;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.promo.mir2020.Mir2020PromoOrder;
import ru.yandex.travel.orders.entities.promo.mir2020.MirPromoOrderEligibility;
import ru.yandex.travel.orders.repository.promo.mir2020.Mir2020PromoOrderRepository;
import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

@Service
@Slf4j
@RequiredArgsConstructor
public class Mir2020PromoService {
    private final MirEligibilityService mirEligibilityService;
    private final Mir2020PromoOrderRepository promoOrderRepository;
    private final Mir2020ConfigurationProperties configurationProperties;
    private final SingleOperationService singleOperationService;

    @TransactionMandatory
    public boolean checkInitialEligibility(Order order) {
        if (!OrderCompatibilityUtils.isHotelOrder(order)) {
            return false;
        }
        HotelOrderItem orderItem = OrderCompatibilityUtils.getOnlyHotelOrderItem(order);
        try {
            var mirEligibility = mirEligibilityService.checkEligibility(orderItem.getPartnerId(),
                    orderItem.getHotelItinerary().getOrderDetails().getOriginalId(),
                    Instant.now(), orderItem.getHotelItinerary().getOrderDetails().getCheckinDate(),
                    orderItem.getHotelItinerary().getOrderDetails().getCheckoutDate(),
                    order.calculateTotalCost().getNumberStripped());
            Mir2020PromoOrder promo;
            var existing = promoOrderRepository.findById(order.getId());
            if (existing.isPresent()) {
                promo = existing.get();
            } else {
                promo = new Mir2020PromoOrder();
                promo.setOrderId(order.getId());
            }

            switch (mirEligibility.getEligibility()) {
                case ME_ELIGIBLE:
                    promo.setEligibility(MirPromoOrderEligibility.ELIGIBLE);
                    promo.setMirHotelId(mirEligibility.getMirId());
                    promo.setCashbackAmount(mirEligibility.getCashbackAmount().getValue());
                    break;
                case ME_WRONG_STAY_DATES:
                    promo.setEligibility(MirPromoOrderEligibility.WRONG_STAY_DATES);
                    break;
                case ME_WRONG_LOS:
                    promo.setEligibility(MirPromoOrderEligibility.WRONG_LOS);
                    break;
                case ME_BLACKLISTED:
                    promo.setEligibility(MirPromoOrderEligibility.BLACKLISTED);
                    break;
                case ME_WRONG_BOOKING_DATE:
                    promo.setEligibility(MirPromoOrderEligibility.PROMO_DISABLED);
                    break;
            }
            if (existing.isEmpty()) {
                promoOrderRepository.save(promo);
            }
            return promo.getEligibility() == MirPromoOrderEligibility.ELIGIBLE;
        } catch (Exception e) {
            log.error("Exception occurred while trying to determine " +
                    "if mir promo for hotel order is allowed, falling back to false", e);
            return false;
        }
    }

    @TransactionMandatory
    public void registerConfirmedOrder(Order order) {
        if (order.getPaymentSchedule() != null) {
            log.info("Deferred payments are never mir-eligible (order {})", order.getId().toString());
            return;
        }
        if (order.isFullyPostPaid()) {
            log.info("Post paid orders are never mir-eligible (order {})", order.getId().toString());
            return;
        }
        Preconditions.checkState(order.getCurrentInvoice() != null, "Expected order to have an invoice");
        Preconditions.checkState(order.getCurrentInvoice().getTrustPaymentTs() != null, "Expected invoice to have a " +
                "payment ts");
        HotelOrderItem item = OrderCompatibilityUtils.getOnlyHotelOrderItem(order);
        try {
            Mir2020PromoOrder promo = promoOrderRepository.getOne(order.getId());
            if (promo.getEligibility() == MirPromoOrderEligibility.ELIGIBLE) {
                if (!mirEligibilityService.isPromoActive(order.getCurrentInvoice().getTrustPaymentTs())) {
                    log.info("Eligible order {} paid after the promo has ended", order.getId().toString());
                    promo.setEligibility(MirPromoOrderEligibility.PROMO_DISABLED);
                } else {
                    if (configurationProperties.getCardTypes().contains(order.getCurrentInvoice().getCardType())) {
                        log.info("Eligible order {} paid with eligible card", order.getId().toString());
                        promo.setPaidWithMir(true);
                    } else {
                        log.info("Eligible order {} paid with non-eligible {} card", order.getId().toString(),
                                order.getCurrentInvoice().getCardType());
                        promo.setPaidWithMir(false);
                        if (configurationProperties.getSupport().isEnabled() && item.getHotelItinerary().getRefundRules()
                                .isFullyRefundableAt(Instant.now().plus(configurationProperties.getSupport().getMinFreeRefundRemaining()))) {
                            log.info("Creating support notification: the order is refundable, so we may persuade the " +
                                    "user" +
                                    " to refund and pay with other card");
                            singleOperationService.scheduleOperation("MirSupportNotification_" + order.getPrettyId(),
                                    OperationTypes.MIR_SUPPORT_NOTIFICATION_SENDER.getValue(), order.getId(),
                                    Instant.now().plus(configurationProperties.getSupport().getNotificationDelay()));
                        }
                    }
                }
                promoOrderRepository.save(promo);
            }
        } catch (Exception ex) {
            log.error("Unable to update mir eligibility", ex);
        }
    }

    @TransactionMandatory
    public Mir2020PromoOrder getPromoParticipationStatus(Order order) {
        return promoOrderRepository.findById(order.getId()).orElse(null);
    }
}
