package ru.yandex.travel.orders.services.promo.taxi2020;

import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoCode;
import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrder;
import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus;
import ru.yandex.travel.orders.repository.HotelOrderRepository;
import ru.yandex.travel.orders.repository.promo.taxi2020.Taxi2020PromoCodeRepository;
import ru.yandex.travel.orders.repository.promo.taxi2020.Taxi2020PromoOrderRepository;
import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.ELIGIBLE;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.EMAIL_SCHEDULED;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.NOT_ELIGIBLE;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.PROMO_CODE_CAN_BE_ASSIGNED;
import static ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils.isConfirmed;

@Service
@RequiredArgsConstructor
@Slf4j
public class Taxi2020PromoService {
    static final String SINGLETON_ORDERS_WAITING_FOR_PROCESSING_TASK = "singletonTaxi2020PromoOrderProcessorTask";

    private final Taxi2020PromoServiceProperties properties;
    private final Taxi2020PromoOrderRepository promoOrderRepository;
    private final Taxi2020PromoCodeRepository promoCodeRepository;
    private final HotelOrderRepository hotelOrderRepository;
    private final Clock clock;
    private final SingleOperationService singleOperationService;

    @VisibleForTesting
    boolean isPromoActive(Instant createdAt) {
        // start <= createdAt && createdAt < end
        return !properties.getStartsAt().isAfter(createdAt) && createdAt.isBefore(properties.getEndsAt());
    }

    @TransactionMandatory
    public void registerConfirmedOrder(HotelOrder order) {
        registerConfirmedOrderImpl(order);
    }

    @TransactionMandatory
    public void registerConfirmedOrder(GenericOrder order) {
        if (!OrderCompatibilityUtils.isHotelOrder(order)) {
            return;
        }
        registerConfirmedOrderImpl(order);
    }

    private void registerConfirmedOrderImpl(Order order) {
        if (isPromoActive(order.getCreatedAt())) {
            Taxi2020PromoOrder promoOrder = new Taxi2020PromoOrder();
            promoOrder.setOrderId(order.getId());
            synchronizePromoOrderStatus(promoOrder, order);
            promoOrderRepository.save(promoOrder);
            log.info("The Taxi 2020 promo is active, start tracking order updates. " +
                            "The current order status is {}, email is {}, scheduled at {}",
                    promoOrder.getStatus(), promoOrder.getEmail(), promoOrder.getEmailScheduledAt());

            // even if the order is ELIGIBLE and the email can be sent right now,
            // we don't assign a promo code here to avoid possible problems with ran out free promo codes,
            // this step will be performed asynchronously by the task processor code below
        } else {
            log.info("The Taxi 2020 promo is not active yet, skipping the order");
        }
    }

    @TransactionMandatory
    public Collection<String> findOrderIdsWaitingForProcessing(Set<String> excludeIds) {
        if (excludeIds.contains(SINGLETON_ORDERS_WAITING_FOR_PROCESSING_TASK)) {
            log.warn("Querying for more tasks while processing {}. " +
                    "It shouldn't happen in case of a single threaded task processor (expected)", excludeIds);
            return List.of();
        }
        if (countOrdersWaitingForProcessing() == 0) {
            return List.of();
        }
        return List.of(SINGLETON_ORDERS_WAITING_FOR_PROCESSING_TASK);
    }

    @TransactionMandatory
    public Long countOrdersWaitingForProcessing() {
        return promoOrderRepository.countOrderToCheck(getCurrentTime());
    }

    @TransactionMandatory
    public void processOrdersBatch(String taskId) {
        Preconditions.checkArgument(taskId.equals(SINGLETON_ORDERS_WAITING_FOR_PROCESSING_TASK),
                "Only the singleton task id is expected but got %s", taskId);

        Pageable batchPage = PageRequest.of(0, properties.getMaxProcessOrdersBatch());
        Instant now = getCurrentTime();
        List<Taxi2020PromoOrder> ordersForProcessing = promoOrderRepository.findOrdersToCheck(now, batchPage);
        if (ordersForProcessing.isEmpty()) {
            log.warn("No actual orders to process");
            return;
        }

        log.info("Checking {} Taxi 2020 promo orders for status change or email readiness", ordersForProcessing.size());
        List<UUID> orderIds = ordersForProcessing.stream().map(Taxi2020PromoOrder::getOrderId).collect(toList());
        Map<UUID, HotelOrder> hotelOrders = hotelOrderRepository.findAllById(orderIds).stream()
                .collect(toMap(HotelOrder::getId, ho -> ho));
        for (Taxi2020PromoOrder order : ordersForProcessing) {
            HotelOrder hotelOrder = hotelOrders.get(order.getOrderId());
            Preconditions.checkNotNull(hotelOrder, "Hotel order {} not found", order.getOrderId());
            try (NestedMdc ignored = NestedMdc.forEntity(hotelOrder)) {
                processPendingOrder(order, hotelOrder);
            }
        }
    }

    @VisibleForTesting
    void processPendingOrder(Taxi2020PromoOrder promoOrder, HotelOrder hotelOrder) {
        log.debug("Checking Taxi 2020 promo order status");
        Taxi2020PromoOrderStatus sourceStatus = promoOrder.getStatus();

        if (promoOrder.getStatus() == ELIGIBLE || promoOrder.getStatus() == NOT_ELIGIBLE) {
            synchronizePromoOrderStatus(promoOrder, hotelOrder);
        }

        if (promoOrder.getStatus() == PROMO_CODE_CAN_BE_ASSIGNED) {
            log.info("Assigning promo code and scheduling Taxi 2020 promo email to {}", promoOrder.getEmail());
            promoOrder.setPromoCode(chooseFreePromoCodeToAssign());
            promoOrder.setStatus(EMAIL_SCHEDULED);
            Taxi2020PromoSendMailData emailData = Taxi2020PromoSendMailData.builder()
                    .orderId(promoOrder.getOrderId())
                    .email(promoOrder.getEmail())
                    .promoCode(promoOrder.getPromoCode())
                    .build();
            UUID operationId = singleOperationService.runOperation(
                    "Taxi2020PromoSendMail" + getCurrentTime().toEpochMilli(),
                    OperationTypes.TAXI_2020_PROMO_CODE_EMAIL_SENDER.getValue(),
                    emailData
            );
            promoOrder.setSendEmailOperationId(operationId);
            Taxi2020PromoSendSmsData smsData = Taxi2020PromoSendSmsData.builder()
                    .orderId(promoOrder.getOrderId())
                    .phone(hotelOrder.getPhone())
                    .promoCode(promoOrder.getPromoCode())
                    .build();
            UUID smsOperationId = singleOperationService.scheduleOperation(
                    "Taxi2020PromoSendSms" + getCurrentTime().toEpochMilli(),
                    OperationTypes.TAXI_2020_PROMO_CODE_SMS_SENDER.getValue(),
                    smsData,
                    getSmsScheduledAt(hotelOrder)
            );
            promoOrder.setSendSmsOperationId(smsOperationId);
        }

        if (sourceStatus != promoOrder.getStatus()) {
            log.info("Taxi 2020 promo order status changed from {} to {}", sourceStatus, promoOrder.getStatus());
        }
    }

    // we call this method on order confirmation and any of its subsequent updates
    private void synchronizePromoOrderStatus(Taxi2020PromoOrder promoOrder, Order hotelOrder) {
        // if a confirmed order doesn't match promo terms anymore but did match them before,
        // we should still send a promo code its owner
        boolean isAlreadyPromised = promoOrder.getStatus() == ELIGIBLE;
        if (promoTermsMatch(hotelOrder) || (isAlreadyPromised && isConfirmed(hotelOrder))) {
            Instant emailScheduledAt = getEmailScheduledAt(hotelOrder);
            // scheduledAt <= now
            boolean isReady = !emailScheduledAt.isAfter(getCurrentTime());
            promoOrder.setStatus(isReady ? PROMO_CODE_CAN_BE_ASSIGNED : ELIGIBLE);
            promoOrder.setEmail(hotelOrder.getEmail());
            promoOrder.setEmailScheduledAt(emailScheduledAt);
        } else {
            promoOrder.setStatus(NOT_ELIGIBLE);
            promoOrder.setEmail(null);
            promoOrder.setEmailScheduledAt(null);
        }

        Instant maxUpdatedAt = hotelOrder.getUpdatedAt();
        for (OrderItem orderItem : hotelOrder.getOrderItems()) {
            if (orderItem.getUpdatedAt().isAfter(maxUpdatedAt)) {
                maxUpdatedAt = orderItem.getUpdatedAt();
            }
        }
        promoOrder.setStatusUpdatedAt(maxUpdatedAt);
    }

    @VisibleForTesting
    boolean promoTermsMatch(Order order) {
        Preconditions.checkArgument(order.getCurrency().getCurrencyCode().equals(properties.getMinPriceCurrency()),
                "Only %s order currency os expected but got %s",
                properties.getMinPriceCurrency(), order.getOrderItems().size());
        HotelOrderItem orderItem = OrderCompatibilityUtils.getOnlyHotelOrderItem(order);
        Money price = order.calculateTotalCost();
        Money minPrice = Money.of(properties.getMinPriceValue(), properties.getMinPriceCurrency());
        LocalDate checkIn = orderItem.getHotelItinerary().getOrderDetails().getCheckinDate();
        // the 'createdAt in [startsAt, endsAt)' condition is checked in registerConfirmedOrder
        return isConfirmed(order)
                && price.isGreaterThanOrEqualTo(minPrice)
                // checkIn <= maxCheckIn
                && !checkIn.isAfter(properties.getMaxCheckInDate());
    }

    private Instant getCurrentTime() {
        return Instant.now(clock);
    }

    private Instant getCheckInDateStart(Order order) {
        HotelOrderItem orderItem = OrderCompatibilityUtils.getOnlyHotelOrderItem(order);
        OrderDetails orderDetails = orderItem.getHotelItinerary().getOrderDetails();
        LocalDate checkIn = orderDetails.getCheckinDate();
        ZoneId hotelTimeZoneId = orderDetails.getHotelTimeZoneId();
        if (hotelTimeZoneId == null) {
            log.warn("No hotel time zone id for permalink id {}, using {} instead for Taxi 2020 promo",
                    orderDetails.getPermalink(), properties.getDefaultHotelTimeZoneId());
            hotelTimeZoneId = properties.getDefaultHotelTimeZoneId();
        }
        return checkIn.atStartOfDay(hotelTimeZoneId).toInstant();
    }

    private Instant getEmailScheduledAt(Order order) {
        Instant checkInDayStart = getCheckInDateStart(order);
        return checkInDayStart.plus(properties.getEmailScheduledAtOffset());
    }

    private Instant getSmsScheduledAt(HotelOrder order) {
        Instant checkInDayStart = getCheckInDateStart(order);
        return checkInDayStart.plus(properties.getSmsScheduledAtOffset());
    }

    private String chooseFreePromoCodeToAssign() {
        Taxi2020PromoCode freePromoCode = promoCodeRepository.findAnyByUsedAtIsNull();
        Instant now = getCurrentTime();
        if (freePromoCode == null) {
            throw new IllegalStateException("No more free promo codes!");
        }
        if (freePromoCode.getExpiresAt().isBefore(now)) {
            // "should never happen"
            throw new IllegalStateException("Expired promo-code: " + freePromoCode.getExpiresAt());
        }
        // no strict locking here as the code will be referenced by a unique column from Taxi2020PromoOrder,
        // in case of multiple transactions only one will manage to store the reference and update usedAt
        freePromoCode.setUsedAt(now);
        return freePromoCode.getCode();
    }

    @TransactionMandatory
    public Taxi2020PromoOrder getPromoParticipationStatus(HotelOrder order) {
        return promoOrderRepository.findById(order.getId()).orElse(null);
    }
}
