package ru.yandex.travel.orders.services.report;

import java.io.ByteArrayOutputStream;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

import com.fasterxml.jackson.databind.node.POJONode;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.net.MediaType;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.Nullable;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.datetime.DateTimeUtils;
import ru.yandex.travel.commons.streams.CustomCollectors;
import ru.yandex.travel.hotels.administrator.export.proto.ContractInfo;
import ru.yandex.travel.orders.cache.BalanceContractDictionary;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.finances.BankOrder;
import ru.yandex.travel.orders.entities.finances.BankOrderDetail;
import ru.yandex.travel.orders.entities.finances.OebsPaymentStatus;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.entities.notifications.AttachmentProviderType;
import ru.yandex.travel.orders.entities.notifications.EmailChannelInfo;
import ru.yandex.travel.orders.entities.notifications.Notification;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.repository.NotificationRepository;
import ru.yandex.travel.orders.repository.finances.BankOrderRepository;
import ru.yandex.travel.orders.services.attachments.AttachmentsHelper;
import ru.yandex.travel.orders.services.cloud.s3.S3Service;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinProperties;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;
import ru.yandex.travel.workflow.single_operation.SingleOperationRunner;

@Service
@EnableConfigurationProperties(HotelsReportProperties.class)
@RequiredArgsConstructor
@Slf4j
public class HotelPartnerPaymentOrderReportSender implements SingleOperationRunner<HotelPartnerPaymentOrderReportSender.Params, UUID> {
    public static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("dd.MM.yyyy");

    private final HotelPartnerReportService reportService;
    private final BalanceContractDictionary balanceContractDictionary;
    private final BankOrderRepository bankOrderRepository;
    private final HotelsReportProperties hotelsReportProperties;
    private final WorkflowRepository workflowRepository;
    private final NotificationRepository notificationRepository;
    private final WorkflowMessageSender workflowMessageSender;
    private final StarTrekService starTrekService;
    private final DolphinProperties dolphinProperties;
    private final S3Service s3Service;


    @Override
    public Class<HotelPartnerPaymentOrderReportSender.Params> getInputClass() {
        return HotelPartnerPaymentOrderReportSender.Params.class;
    }

    @Override
    public UUID runOperation(HotelPartnerPaymentOrderReportSender.Params params) {
        BankOrder bankOrder = bankOrderRepository.findByPaymentBatchIdAndBankOrderId(params.paymentBatchId,
                params.getBankOrderId());
        Preconditions.checkState(bankOrder != null, "Bank order not found");

        if (bankOrder.getOebsPaymentStatus() == OebsPaymentStatus.RETURNED) {
            log.warn("Payment order {} of batch {} is returned, will notify accounting team instead of sending report",
                    params.getBankOrderId(), params.getPaymentBatchId());
            return starTrekService.createIssueAboutReturnedPaymentOrder(bankOrder, params.paymentBatchId,
                    workflowMessageSender);
        } else {
            Preconditions.checkState(!bankOrder.getBankOrderPayment().getDetails().isEmpty(),
                    "Empty bank order details on payment report");
            Long contractId = bankOrder.getBankOrderPayment().getDetails().stream()
                    .map(BankOrderDetail::getContractId)
                    .distinct()
                    .collect(CustomCollectors.exactlyOne());
            if (contractId.equals(dolphinProperties.getFinancialData().getBillingContractId())) {
                log.info("Ignoring Dolphin payment order: we should not send its report");
                return null;
            }
            ContractInfo contractInfo = balanceContractDictionary.findContractInfoByContractId(contractId);
            Preconditions.checkArgument(contractInfo != null, "Contract %s not found", contractId);

            if (!hotelsReportProperties.getMail().isSendPaymentOrdersReport()) {
                log.info("Payment report sending is disabled");
                return null;
            }
            log.info("Will send a payment order report for order {} of batch {}", params.getBankOrderId(), params.getPaymentBatchId());
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            boolean reportIsNotEmpty = reportService.writePaymentOrderReportToOutputStream(params.bankOrderId,
                    params.paymentBatchId, stream);
            if (!reportIsNotEmpty) {
                log.info("Report is empty");
                return null;
            }
            var emailInfo = new EmailChannelInfo();
            emailInfo.setCampaign(hotelsReportProperties.getMail().getPaymentOrdersReportCampaign());

            String subjectLegalPart = contractInfo.getFullLegalName();
            List<String> targets = getTargets(params, contractInfo);
            List<String> bccTargets = hotelsReportProperties.getMail().getBccTargets();
            if (targets == null || targets.size() == 0) {
                targets = bccTargets;
                bccTargets = Collections.emptyList();
                subjectLegalPart += " [!!! email не указан !!!]";
            }
            emailInfo.setMultipleTargets(targets);
            emailInfo.setBccTargets(bccTargets);

            String bankOrderId = bankOrder.getBankOrderId();
            String orderDate = bankOrder.getEventtime().format(DATE_TIME_FORMATTER);
            ReportMailData mailData = new ReportMailData();
            mailData.setOrderId(bankOrderId);
            mailData.setOrderDate(orderDate);
            mailData.setSubjectLegalPart(subjectLegalPart);
            mailData.setSubjectContractPart(contractInfo.getExternalContractId());
            mailData.setContractConfirmDate(ReportDateHelper.getDayMonthYearStandard(DateTimeUtils.standardParseLocalDate(contractInfo.getExternalContractRegisterDate())));
            emailInfo.setArguments(new POJONode(mailData));
            Notification email = Notification.createEmailNotification(emailInfo);
            Workflow workflow = Workflow.createWorkflowForEntity(email,
                    WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
            workflow = workflowRepository.save(workflow);
            var filename = String.format("Реестр бронирований по ПП №%s от %s.xlsx", bankOrderId, orderDate);
            var bookingsReport = Attachment.createReadyAttachment(email, filename,
                    MediaType.OOXML_SHEET.toString(), AttachmentProviderType.HOTEL_BOOKING_REPORT);
            AttachmentsHelper.storeAttachmentData(bookingsReport, stream.toByteArray(), s3Service);
            notificationRepository.save(email);
            workflowMessageSender.scheduleEvent(workflow.getId(), TSend.getDefaultInstance());
            log.info("Email notification {} scheduled", email.getId());
            return email.getId();
        }
    }

    @Nullable
    private List<String> getTargets(Params params, ContractInfo contractInfo) {
        if (!Strings.isNullOrEmpty(params.getEmail())) {
            log.info("Email {} passed with the event, will override the partner's default", params.getEmail());
            return List.of(params.getEmail());
        }
        List<String> targets = null;
        if (Strings.isNullOrEmpty(hotelsReportProperties.getMail().getEmailOverride())) {
            if (!Strings.isNullOrEmpty(contractInfo.getAccountantEmails())) {
                targets = List.of(contractInfo.getAccountantEmails().split(";"));
            } else {
                log.warn("Email not specified for client id {}, contract id {}", contractInfo.getClientId(), contractInfo.getContractId());
            }
        } else {
            log.warn("Email override is specified in config: all reports are routed to {}", hotelsReportProperties.getMail().getEmailOverride());
            targets = List.of(hotelsReportProperties.getMail().getEmailOverride());
        }
        return targets;
    }


    @Data
    public static final class Params {
        String bankOrderId;
        String paymentBatchId;
        /**
         * Used to override for dev purposes
         */
        String email;

        public static Params fromBankOrder(BankOrder bankOrder) {
            Params input = new Params();
            input.setBankOrderId(bankOrder.getBankOrderId());
            input.setPaymentBatchId(bankOrder.getBankOrderPayment().getPaymentBatchId());
            return input;
        }
    }
}
