package ru.yandex.travel.orders.services.report;

import java.io.ByteArrayOutputStream;
import java.time.Instant;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import com.fasterxml.jackson.databind.node.POJONode;
import com.google.common.base.Strings;
import com.google.common.net.MediaType;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.datetime.DateTimeUtils;
import ru.yandex.travel.hotels.administrator.export.proto.ContractInfo;
import ru.yandex.travel.orders.cache.BalanceContractDictionary;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.entities.notifications.AttachmentProviderType;
import ru.yandex.travel.orders.entities.notifications.EmailChannelInfo;
import ru.yandex.travel.orders.entities.notifications.Notification;
import ru.yandex.travel.orders.repository.NotificationRepository;
import ru.yandex.travel.orders.services.attachments.AttachmentsHelper;
import ru.yandex.travel.orders.services.cloud.s3.S3Service;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;
import ru.yandex.travel.workflow.single_operation.SingleOperationRunner;

/**
 * Operation return the uuid of notification sent to partner
 */
@Service
@EnableConfigurationProperties(HotelsReportProperties.class)
@RequiredArgsConstructor
@Slf4j
public class HotelPartnerReportSender implements SingleOperationRunner<HotelPartnerReportSender.Params, UUID> {

    private static final DateTimeFormatter FILE_NAME_FORMATTER = DateTimeFormatter.ofPattern("MM yyyy");
    private static final DateTimeFormatter CONTRACT_REGISTER_DATE_FORMATTER = DateTimeFormatter.ofPattern("dd.MM.yyyy");

    @Data
    public static final class Params {
        Long billingClientId;
        LocalDate periodBegin;
        LocalDate periodEnd;
        String email;
        Instant reportAt;
    }

    @Data
    public static final class PartnerLegalInfo {
        String fullLegalName;
        String externalContractId;
        Set<String> emails = new HashSet<>();
        boolean sendEmptyOrdersReport;
        LocalDate externalContractRegisteredAt;
    }

    private final HotelPartnerReportService hotelPartnerReportService;

    private final HotelsReportProperties hotelsReportProperties;

    private final BalanceContractDictionary balanceContractDictionary;

    private final NotificationRepository notificationRepository;

    private final WorkflowRepository workflowRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final S3Service s3Service;

    @Override
    public Class<Params> getInputClass() {
        return Params.class;
    }

    @Override
    public UUID runOperation(Params params) {
        ByteArrayOutputStream actedOrdersOutput = new ByteArrayOutputStream();
        PartnerLegalInfo partnerLegalInfo = getPartnerLegalInfo(balanceContractDictionary.findContractInfoByClientId(params.getBillingClientId()));
        if (Strings.isNullOrEmpty(params.getEmail()) && partnerLegalInfo.getEmails().size() == 0) {
            // Ignore partners with no e-mail
            return null;
        }

        var ordersReportIsNotEmpty = hotelPartnerReportService.writeActedOrdersReportToOutputStream(params.getBillingClientId(),
                partnerLegalInfo.getFullLegalName(), params.getPeriodBegin(), params.getPeriodEnd(), actedOrdersOutput);

        if ((!ordersReportIsNotEmpty && !partnerLegalInfo.isSendEmptyOrdersReport()) ||
                !hotelsReportProperties.getMail().isSendAccountingAct()) {
            log.info("Do not send e-mail as orders report is empty");
            return null;
        }

        var emailInfo = new EmailChannelInfo();
        emailInfo.setCampaign(hotelsReportProperties.getMail().getOrdersReportCampaign());
        emailInfo.setArguments(new POJONode(createReportMailData(params, partnerLegalInfo)));
        if (!Strings.isNullOrEmpty(params.getEmail())) {
            emailInfo.setMultipleTargets(List.of(params.getEmail()));
        } else {
            emailInfo.setMultipleTargets(List.copyOf(partnerLegalInfo.getEmails()));
        }
        emailInfo.setBccTargets(hotelsReportProperties.getMail().getBccTargets());
        var email = Notification.createEmailNotification(emailInfo);

        Workflow workflow = Workflow.createWorkflowForEntity(email,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);
        String datePart = ReportDateHelper.getMonthYearText(params.getPeriodBegin());

        if (hotelsReportProperties.getMail().isSendAccountingAct()) {
            var actedOrdersReport = Attachment.createReadyAttachment(email,
                    "Реестр завершенных бронирований за " + datePart + ".xlsx",
                    MediaType.OOXML_SHEET.toString(), AttachmentProviderType.HOTEL_ACCOUNTING_ACT);
            AttachmentsHelper.storeAttachmentData(actedOrdersReport, actedOrdersOutput.toByteArray(), s3Service);
        }

        notificationRepository.save(email);
        workflowMessageSender.scheduleEvent(workflow.getId(), TSend.getDefaultInstance());

        return email.getId();
    }

    private PartnerLegalInfo getPartnerLegalInfo(ContractInfo contractInfo) {
        var partnerLegalInfo = new PartnerLegalInfo();
        if (contractInfo == null) {
            return partnerLegalInfo;
        }

        if (!Strings.isNullOrEmpty(contractInfo.getAccountantEmails())) {
            Collections.addAll(partnerLegalInfo.getEmails(), contractInfo.getAccountantEmails().split(";"));
        }
        if (!Strings.isNullOrEmpty(contractInfo.getExternalContractId())) {
            partnerLegalInfo.setExternalContractId(contractInfo.getExternalContractId());
        }
        if (!Strings.isNullOrEmpty(contractInfo.getFullLegalName())) {
            partnerLegalInfo.setFullLegalName(contractInfo.getFullLegalName());
        }
        if (!Strings.isNullOrEmpty(contractInfo.getExternalContractRegisterDate())) {
            partnerLegalInfo.setExternalContractRegisteredAt(DateTimeUtils.standardParseLocalDate(contractInfo.getExternalContractRegisterDate()));
        }
        partnerLegalInfo.setSendEmptyOrdersReport(contractInfo.getSendEmptyOrdersReport());
        return partnerLegalInfo;
    }

    private ReportMailData createReportMailData(Params params, PartnerLegalInfo partnerLegalInfo) {
        ReportMailData result = new ReportMailData();
        result.setSubjectLegalPart(partnerLegalInfo.getFullLegalName());
        result.setSubjectContractPart(partnerLegalInfo.getExternalContractId());
        result.setSubjectDatePart(ReportDateHelper.getMonthYearText(params.getPeriodBegin()));
        if (partnerLegalInfo.getExternalContractRegisteredAt() != null) {
            result.setContractConfirmDate(ReportDateHelper.getDayMonthYearStandard(partnerLegalInfo.getExternalContractRegisteredAt()));
        }
        return result;
    }
}
