package ru.yandex.travel.orders.services.report;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;

import lombok.extern.slf4j.Slf4j;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.JobExecutionContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.quartz.QuartzJobBean;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

@Slf4j
@DisallowConcurrentExecution
public class HotelPartnerReportSenderJob extends QuartzJobBean {
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("MM-yyyy");
    @Autowired
    private SingleOperationService singleOperationService;
    @Autowired
    private TransactionTemplate transactionTemplate;

    @Override
    protected void executeInternal(JobExecutionContext context) {
        LocalDate previousMonth = LocalDate.now().minus(1, ChronoUnit.MONTHS);
        HotelPartnerReportSenderPlanner.Params input = new HotelPartnerReportSenderPlanner.Params();
        input.setPeriodBegin(previousMonth.with(TemporalAdjusters.firstDayOfMonth()));
        input.setPeriodEnd(previousMonth.with(TemporalAdjusters.lastDayOfMonth()));
        input.setReportType("orders");
        transactionTemplate.execute(unused -> {
            var operationId = singleOperationService.runUniqueOperation(
                    "PlanSendingHotelReports " + LocalDate.now().format(DATE_FORMATTER),
                    OperationTypes.HOTELS_PLAN_SENDING_REPORTS_TO_PARTNERS.getValue(),
                    input);
            log.debug("Scheduled PlanSendingHotelReports: " + operationId.toString());
            return operationId;
        });
        HotelPartnerReportSenderPlanner.Params payoutInput = new HotelPartnerReportSenderPlanner.Params();
        payoutInput.setPeriodBegin(previousMonth.with(TemporalAdjusters.firstDayOfMonth()));
        payoutInput.setPeriodEnd(previousMonth.with(TemporalAdjusters.lastDayOfMonth()));
        payoutInput.setReportType("payout");
        transactionTemplate.execute(unused -> {
            var operationId = singleOperationService.runUniqueOperation(
                    "PlanSendingHotelPayoutReports " + LocalDate.now().format(DATE_FORMATTER),
                    OperationTypes.HOTELS_PLAN_SENDING_REPORTS_TO_PARTNERS.getValue(),
                    payoutInput);
            log.debug("Scheduled PlanSendingHotelPayoutReports: " + operationId.toString());
            return operationId;
        });
    }
}
