package ru.yandex.travel.orders.services.report;

import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.cache.HotelAgreementDictionary;
import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.orders.services.finances.billing.BillingHelper;
import ru.yandex.travel.workflow.single_operation.SingleOperationRunner;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

@Service
@RequiredArgsConstructor
@Slf4j
public class HotelPartnerReportSenderPlanner implements SingleOperationRunner<HotelPartnerReportSenderPlanner.Params, Map<Long, UUID>> {

    private final HotelAgreementDictionary hotelAgreementDictionary;
    private final SingleOperationService singleOperationService;
    private final Clock clock;
    private final static Random randomizer = new Random();

    @Data
    public static final class Params {
        LocalDate periodBegin;
        LocalDate periodEnd;
        String reportType;
    }

    @Override
    public Class<Params> getInputClass() {
        return Params.class;
    }

    @Override
    public Map<Long, UUID> runOperation(Params params) {
        Set<Long> partnerIds = hotelAgreementDictionary.getAllFinancialClientsForPartnersInGivenPeriod(
                Set.of(EPartnerId.PI_TRAVELLINE, EPartnerId.PI_BNOVO),
                BillingHelper.getBillingDayStart(params.getPeriodBegin()),
                BillingHelper.getBillingDayEnd(params.getPeriodEnd())
        );
        String operationType;
        if ("payout".equals(params.getReportType())) {
            operationType = OperationTypes.HOTELS_SEND_PARTNER_PAYOUT_ORDER.getValue();
        } else {
            operationType = OperationTypes.HOTELS_SEND_PARTNER_PAYOUT_DETAILS_REPORT.getValue();
        }
        return partnerIds.stream().collect(Collectors.toMap(
                partnerId -> partnerId,
                partnerId -> {
                    HotelPartnerReportSender.Params input = new HotelPartnerReportSender.Params();
                    input.setBillingClientId(partnerId);
                    input.setPeriodBegin(params.getPeriodBegin());
                    input.setPeriodEnd(params.getPeriodEnd());
                    input.setReportAt(Instant.now(clock));
                    Instant scheduleAt = Instant.now(clock).plusSeconds(randomizer.nextInt(172800));
                    return singleOperationService.scheduleUniqueOperationUnsafe(
                            "SendPartnerReport:" + partnerId + " " + Instant.now(clock).toEpochMilli(),
                            operationType,
                            input,
                            scheduleAt);
                }));
    }
}
