package ru.yandex.travel.orders.services.report;

import java.io.OutputStream;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.datetime.DateTimeUtils;
import ru.yandex.travel.commons.streams.CustomCollectors;
import ru.yandex.travel.orders.cache.BalanceContractDictionary;
import ru.yandex.travel.orders.entities.finances.BankOrder;
import ru.yandex.travel.orders.entities.finances.BankOrderDetail;
import ru.yandex.travel.orders.repository.finances.BankOrderDetailRepository;
import ru.yandex.travel.orders.repository.finances.BankOrderRepository;
import ru.yandex.travel.orders.services.report.model.PartnerOrdersReport;
import ru.yandex.travel.orders.services.report.model.PartnerPaymentOrdersReport;
import ru.yandex.travel.orders.services.report.model.PartnerPayoutsReport;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Slf4j
@Service
@RequiredArgsConstructor
public class HotelPartnerReportService {

    private final static DateTimeFormatter sheetNameFormatter = DateTimeFormatter.ofPattern("yyyy-MM");
    private final static DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("dd.MM.yyyy");

    private final BankOrderRepository bankOrderRepository;
    private final BankOrderDetailRepository bankOrderDetailRepository;
    private final PartnerXlsReportGenerator partnerXlsReportGenerator;
    private final PartnerTransactionsFetcher partnerTransactionsFetcher;
    private final BalanceContractDictionary balanceContractDictionary;

    @TransactionMandatory
    public boolean writePayoutReportToOutputStream(Long billingClientId,
                                                   Long billingContractId,
                                                   String externalContractId,
                                                   LocalDate externalContractDate,
                                                   String billingClientFullName,
                                                   LocalDate from,
                                                   LocalDate to,
                                                   OutputStream out) {
        PartnerPayoutsReport reportData = new PartnerPayoutsReport();
        String datePart = ReportDateHelper.getMonthYearText(from);
        String reportName = "Реестр бронирований по отчету агента за " + datePart;
        if (!Strings.isNullOrEmpty(billingClientFullName)) {
            reportName = reportName + " для " + billingClientFullName;
        }
        reportData.setReportName(reportName);
        if (!Strings.isNullOrEmpty(externalContractId)) {
            String contractData = "Договор №" + externalContractId;
            if (externalContractDate != null) {
                contractData = contractData + " от " + ReportDateHelper.getDayMonthYearStandard(externalContractDate);
            }
            reportData.setContractData(contractData);
        }
        reportData.setReportPeriodFrom(from);
        reportData.setReportPeriodTo(to);

        reportData.setBalanceAtStartOfMonth(bankOrderDetailRepository.getBalanceByContractIdAtDate(billingContractId,
                from));
        reportData.setBalanceAtEndOfMonth(bankOrderDetailRepository.getBalanceByContractIdAtDate(billingContractId,
                to));
        if (reportData.getBalanceAtStartOfMonth() != null && reportData.getBalanceAtEndOfMonth() != null) {
            reportData.setTransferAmount(reportData.getBalanceAtEndOfMonth() - reportData.getBalanceAtStartOfMonth());
        }
        reportData.setTransferredAmount(bankOrderRepository.getReconciledAmountForContractIdInPeriod(billingContractId, from, to));

        reportData.setTransactions(
                partnerTransactionsFetcher.fetchPartnerPayoutTransactions(billingClientId, from, to)
        );

        String sheetDatePart = sheetNameFormatter.format(from);
        partnerXlsReportGenerator.generatePayoutReport("Отчет Агента " + sheetDatePart, reportData, out);

        return reportData.getTransactions().size() > 0;
    }

    @TransactionMandatory
    public boolean writeActedOrdersReportToOutputStream(Long billingClientId, String billingClientFullName,
                                                        LocalDate from, LocalDate to,
                                                        OutputStream out) {
        PartnerOrdersReport reportData = new PartnerOrdersReport();
        String datePart = ReportDateHelper.getMonthYearText(from);
        String reportName = "Реестр завершенных бронирований по акту об исполнении поручения за " + datePart;
        if (!Strings.isNullOrEmpty(billingClientFullName)) {
            reportName = reportName + " для " + billingClientFullName;
        }
        reportData.setReportName(reportName);
        reportData.setReportPeriodFrom(from);
        reportData.setReportPeriodTo(to);
        reportData.setTransactions(
                partnerTransactionsFetcher.fetchPartnerOrderTransactions(billingClientId, from, to)
        );

        String sheetDatePart = sheetNameFormatter.format(from);
        partnerXlsReportGenerator.generateOrdersReport("Реестр заказов " + sheetDatePart, reportData, out);

        return reportData.getTransactions().size() > 0;
    }

    @TransactionMandatory
    public boolean writePaymentOrderReportToOutputStream(String bankOrderId, String paymentBatchId, OutputStream out) {
        BankOrder bankOrder = bankOrderRepository.findByPaymentBatchIdAndBankOrderId(paymentBatchId, bankOrderId);
        Preconditions.checkArgument(bankOrder != null, "Bank Order is not found");
        Long contractId = bankOrder.getBankOrderPayment().getDetails().stream()
                .map(BankOrderDetail::getContractId)
                .distinct()
                .collect(CustomCollectors.exactlyOne());
        var contractInfo = balanceContractDictionary.findContractInfoByContractId(contractId);
        Preconditions.checkArgument(contractInfo != null, "Contract %s not found", contractId);
        PartnerPaymentOrdersReport reportData = new PartnerPaymentOrdersReport();
        reportData.setReportName(String.format("Реестр бронирований по ПП № %s от %s для %s",
                bankOrderId,
                bankOrder.getEventtime().format(dateFormatter),
                contractInfo.getFullLegalName()));
        reportData.setContractData(String.format("Договор №%s от %s", contractInfo.getExternalContractId(),
                ReportDateHelper.getDayMonthYearStandard(DateTimeUtils.standardParseLocalDate(contractInfo.getExternalContractRegisterDate()))));
        reportData.setBankOrderDescription(bankOrder.getDescription());
        reportData.setTransactions(partnerTransactionsFetcher.fetchPartnerPaymentOrderTransactions(bankOrder));
        partnerXlsReportGenerator.generatePaymentOrdersReport(
                String.format("Пп №%s от %s", bankOrderId, bankOrder.getEventtime().format(dateFormatter)),
                reportData,
                out);
        return reportData.getTransactions().size() > 0;
    }

}
