package ru.yandex.travel.orders.services.report;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.persistence.EntityManager;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.streams.CustomCollectors;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.WellKnownOrderItemDiscriminator;
import ru.yandex.travel.orders.entities.finances.BankOrder;
import ru.yandex.travel.orders.entities.finances.BankOrderDetail;
import ru.yandex.travel.orders.entities.finances.BankOrderStatus;
import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.BillingTransactionIdProjection;
import ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType;
import ru.yandex.travel.orders.entities.finances.BillingTransactionType;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreementProvider;
import ru.yandex.travel.orders.repository.BillingTransactionRepository;
import ru.yandex.travel.orders.repository.finances.BankOrderDetailRepository;
import ru.yandex.travel.orders.services.finances.billing.BillingHelper;
import ru.yandex.travel.orders.services.report.model.CommonPartnerReportDTO;
import ru.yandex.travel.orders.services.report.model.PartnerOrdersReport;
import ru.yandex.travel.orders.services.report.model.PartnerPaymentOrdersReport;
import ru.yandex.travel.orders.services.report.model.PartnerPayoutsReport;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Slf4j
@RequiredArgsConstructor
@Service
public class PartnerTransactionsFetcher {
    private static final Collection<BillingTransactionPaymentType> PARTNER_MONEY_BILLING_TYPES = EnumSet.of(
            BillingTransactionPaymentType.COST,
            BillingTransactionPaymentType.YANDEX_ACCOUNT_COST_WITHDRAW);

    private static final Collection<BillingTransactionPaymentType> YANDEX_MONEY_BILLING_TYPES = EnumSet.of(
            BillingTransactionPaymentType.REWARD,
            BillingTransactionPaymentType.YANDEX_ACCOUNT_REWARD_WITHDRAW);

    private final BillingTransactionRepository billingTransactionRepository;
    private final BankOrderDetailRepository bankOrderDetailsRepository;
    private final EntityManager entityManager;

    @TransactionMandatory
    public List<PartnerPayoutsReport.TransactionRow> fetchPartnerPayoutTransactions(
            Long billingClientId, LocalDate from, LocalDate to
    ) {
        Map<Long, List<BillingTransactionIdProjection>> billingTxEventGrouped =
                billingTransactionRepository.billingTransactionsForPayoutReport(billingClientId,
                                BillingHelper.getBillingDayStart(from),
                                BillingHelper.getBillingDayEnd(to)
                        ).stream()
                        .collect(Collectors.groupingBy(tx -> tx.getSourceFinancialEvent().getId()));

        List<PartnerPayoutsReport.TransactionRow> result = new ArrayList<>();
        for (Map.Entry<Long, List<BillingTransactionIdProjection>> entry : billingTxEventGrouped.entrySet()) {
            List<Long> transactionIds = entry.getValue().stream()
                    .map(BillingTransactionIdProjection::getId)
                    .collect(Collectors.toList());
            List<BillingTransaction> transactions = billingTransactionRepository.findAllById(transactionIds);
            PartnerPayoutsReport.TransactionRow row = new PartnerPayoutsReport.TransactionRow();
            row.setPartnerAmount(0.0);
            row.setFeeAmount(0);

            boolean commonFieldsInit = false;
            boolean positive = true;
            for (BillingTransaction t : transactions) {
                if (!commonFieldsInit) {
                    OrderItem orderItem = t.getSourceFinancialEvent().getOrderItem();
                    fillRowFromItinerary(row, orderItem);
                    switch (t.getTransactionType()) {
                        case PAYMENT:
                            positive = true;
                            row.setTxType("Оплата");
                            break;
                        case REFUND:
                            positive = false;
                            row.setTxType("Возврат");
                            break;
                    }
                    row.setTxDate(BillingHelper.toBillingDate(t.getPayoutAt()).toLocalDate());
                    commonFieldsInit = true;
                }
                BigDecimal value = t.getValue().getNumberStripped();
                if (!positive) {
                    value = value.negate();
                }
                if (PARTNER_MONEY_BILLING_TYPES.contains(t.getPaymentType())) {
                    double partnerAmount = row.getPartnerAmount();
                    row.setPartnerAmount(partnerAmount + value.doubleValue());
                    if (t.getYtId() != null) {
                        Optional<BankOrder> bankOrder = bankOrderDetailsRepository.findAllByYtId(t.getYtId())
                                .stream()
                                .map(BankOrderDetail::getBankOrderPayment)
                                .flatMap(payment -> payment.getOrders().stream())
                                .filter(order -> order.getStatus() == BankOrderStatus.DONE)
                                .findAny();
                        bankOrder.ifPresent(order -> {
                            row.setPaymentOrderNumber(order.getBankOrderId());
                            row.setPaymentDate(order.getEventtime());
                        });
                    }
                } else if (YANDEX_MONEY_BILLING_TYPES.contains(t.getPaymentType())) {
                    double feeAmount = row.getFeeAmount();
                    row.setFeeAmount(feeAmount + value.doubleValue());
                }
            }
            double totalAmount = row.getPartnerAmount() + row.getFeeAmount();
            if (positive) {
                row.setTotalAmount(totalAmount);
            } else {
                row.setTotalRefundAmount(totalAmount);
            }
            result.add(row);
            transactions.forEach(entityManager::detach);
        }
        result.sort(Comparator.comparing(PartnerPayoutsReport.TransactionRow::getTxDate));
        return result;
    }

    private HotelItinerary fillRowFromItinerary(CommonPartnerReportDTO.CommonTransactionRow row, OrderItem orderItem) {
        HotelItinerary hotelItinerary;
        if (WellKnownOrderItemDiscriminator.ORDER_ITEM_TRAVELLINE.equals(orderItem.getType())) {
            var travellineHotelItinerary = ((TravellineOrderItem) orderItem).getItinerary();
            row.setPartnerId(travellineHotelItinerary.getTravellineNumber());
            hotelItinerary = travellineHotelItinerary;
        } else if (WellKnownOrderItemDiscriminator.ORDER_ITEM_BNOVO.equals(orderItem.getType())) {
            var bNovoHotelItinerary = ((BNovoOrderItem) orderItem).getItinerary();
            row.setPartnerId(bNovoHotelItinerary.getBNovoNumber());
            hotelItinerary = bNovoHotelItinerary;
        } else {
            throw new IllegalArgumentException(String.format("Unable to fetch payout transactions for " +
                    "partner %s, UNIMPLEMENTED", orderItem.getType()));
        }

        row.setHotelName(hotelItinerary.getOrderDetails().getHotelName());
        row.setPrettyId(orderItem.getOrder().getPrettyId());
        row.setGuestName(hotelItinerary.getGuests().get(0).getFullNameReversed());
        row.setBookedAt(BillingHelper.toBillingDate(orderItem.getConfirmedAt()).toLocalDate());
        row.setCheckIn(hotelItinerary.getOrderDetails().getCheckinDate());
        row.setCheckOut(hotelItinerary.getOrderDetails().getCheckoutDate());
        return hotelItinerary;
    }

    /**
     * Реестр завершенных бронирований
     */
    @TransactionMandatory
    public List<PartnerOrdersReport.TransactionRow> fetchPartnerOrderTransactions(
            Long billingClientId, LocalDate from, LocalDate to
    ) {
        Preconditions.checkArgument(from.isBefore(to), "The from date should always be less than the to date");
        Map<String, List<BillingTransactionIdProjection>> billingTxOrderGrouped =
                billingTransactionRepository.billingTransactionsForOrdersReport(billingClientId,
                                BillingHelper.getBillingDayStart(from),
                                BillingHelper.getBillingDayStart(to)).stream()
                        .collect(Collectors.groupingBy(BillingTransactionIdProjection::getServiceOrderId));

        List<PartnerOrdersReport.TransactionRow> result = new ArrayList<>();
        for (Map.Entry<String, List<BillingTransactionIdProjection>> entry : billingTxOrderGrouped.entrySet()) {
            List<Long> transactionIds = entry.getValue().stream()
                    .map(BillingTransactionIdProjection::getId)
                    .collect(Collectors.toList());
            List<BillingTransaction> transactions = billingTransactionRepository.findAllById(transactionIds);

            PartnerOrdersReport.TransactionRow row = new PartnerOrdersReport.TransactionRow();
            row.setPartnerAmount(0.0);
            row.setFeeAmount(0);

            boolean commonFieldsInit = false;
            boolean payment = true;

            Money partnerAmount = Money.zero(ProtoCurrencyUnit.RUB);
            Money feeAmount = Money.zero(ProtoCurrencyUnit.RUB);

            for (BillingTransaction t : transactions) {
                if (!commonFieldsInit) {
                    OrderItem orderItem = t.getSourceFinancialEvent().getOrderItem();
                    fillRowFromItinerary(row, orderItem);
                    if (orderItem instanceof DirectHotelBillingPartnerAgreementProvider) {
                        // order CONFIRMED rate is always the same as order REFUNDED rate ATM
                        row.setCommissionRate(
                                ((DirectHotelBillingPartnerAgreementProvider) orderItem).getAgreement()
                                        .getOrderConfirmedRate()
                        );
                    } else {
                        log.warn("Unexpected behavior, orders report is generated for not DBoY partner!");
                    }
                    commonFieldsInit = true;
                }

                Money value = t.getValue();
                if (t.getTransactionType() == BillingTransactionType.REFUND) {
                    payment = false;
                    value = value.negate();
                }

                switch (t.getPaymentType()) {
                    case COST:
                    case YANDEX_ACCOUNT_COST_WITHDRAW:
                        partnerAmount = partnerAmount.add(value);
                        break;
                    case REWARD:
                    case YANDEX_ACCOUNT_REWARD_WITHDRAW:
                        feeAmount = feeAmount.add(value);
                        break;
                    default:
                        throw new IllegalStateException(String.format("Don't know how to handle payment type %s",
                                t.getPaymentType())
                        );
                }
            }

            row.setPartnerAmount(partnerAmount.getNumberStripped().doubleValue());
            row.setFeeAmount(feeAmount.getNumberStripped().doubleValue());
            // total amount here will mean here real money, not the original price (different in case of partial refund)
            row.setTotalAmount(row.getFeeAmount() + row.getPartnerAmount());
            if (payment) {
                row.setType("Бронирование");
            } else {
                row.setType("Штраф за возврат");
            }

            if (!partnerAmount.isZero() || !feeAmount.isZero()) {
                result.add(row);
            }
            transactions.forEach(entityManager::detach);
        }
        result.sort(Comparator.comparing(PartnerOrdersReport.TransactionRow::getCheckOut));
        return result;
    }

    @TransactionMandatory
    public List<PartnerPaymentOrdersReport.TransactionRow> fetchPartnerPaymentOrderTransactions(BankOrder bankOrder) {
        Map<Long, BigDecimal> billingTransactionYtIdToAmountMap = bankOrder.getBankOrderPayment()
                .getDetails().stream()
                .filter(d -> PARTNER_MONEY_BILLING_TYPES.contains(d.getPaymentType()))
                .collect(Collectors.toMap(BankOrderDetail::getYtId, BankOrderDetail::getSum));


        return billingTransactionRepository.findAllByYtIdInAndPaymentTypeIn(billingTransactionYtIdToAmountMap.keySet(),
                        PARTNER_MONEY_BILLING_TYPES)
                .stream()
                .collect(
                        Collectors.groupingBy(bt -> Tuple2.tuple(bt.getSourceFinancialEvent().getOrderItem(),
                                        bt.getTransactionType()),
                                CustomCollectors.summingBigDecimal(bt -> billingTransactionYtIdToAmountMap.get(bt.getYtId()))))
                .entrySet().stream()
                .map(mapEntry -> {
                            OrderItem orderItem = mapEntry.getKey().get1();
                            BillingTransactionType transactionType = mapEntry.getKey().get2();
                            BigDecimal paidAmount = mapEntry.getValue();
                            var row = new PartnerPaymentOrdersReport.TransactionRow();
                            var hotelItinerary = fillRowFromItinerary(row, orderItem);

                            row.setTotalAmount(hotelItinerary.getRealHotelPrice().getNumberStripped().doubleValue());
                            switch (transactionType) {
                                case PAYMENT:
                                    row.setTxType("Оплата");
                                    row.setPartnerAmount(paidAmount.doubleValue());
                                    break;
                                case REFUND:
                                    row.setTxType("Возврат");
                                    row.setPartnerAmount(-1 * paidAmount.doubleValue());
                                    break;
                            }
                            return row;
                        }
                ).collect(Collectors.toList());
    }
}
