package ru.yandex.travel.orders.services.suburban;

import java.time.Duration;
import java.time.Instant;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.FiscalReceiptType;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.entities.notifications.EmailChannelInfo;
import ru.yandex.travel.orders.entities.notifications.Notification;
import ru.yandex.travel.orders.repository.NotificationRepository;
import ru.yandex.travel.orders.services.NotificationFileTypes;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnvProvider;
import ru.yandex.travel.orders.services.suburban.providers.SuburbanProviderBase;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.orders.workflows.orderitem.suburban.SuburbanProperties;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static ru.yandex.travel.orders.entities.WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR;


@Service
@Slf4j
@RequiredArgsConstructor
@EnableConfigurationProperties({SuburbanProperties.class})
public class SuburbanNotificationHelper {
    private final SuburbanOrderItemEnvProvider suburbanEnvProvider;
    private final WorkflowRepository workflowRepository;
    private final NotificationRepository notificationRepository;

    @TransactionMandatory
    public UUID createWorkflowForSuburbanOrderConfirmedEmail(Order order) {
        Preconditions.checkArgument(OrderCompatibilityUtils.isSuburbanOrder(order),
                "Suburban order is expected but got %s", order.getClass().getName());
        Preconditions.checkState(OrderCompatibilityUtils.isConfirmed(order), "Order must be confirmed for " +
                "successful notification");

        Notification email = createEmail(order);
        Workflow workflow = Workflow.createWorkflowForEntity(email, GENERIC_ERROR_SUPERVISOR.getUuid());
        workflow = workflowRepository.save(workflow);

        Attachment trustCheck = createFiscalReceiptAttachment(email, order);
        workflowRepository.save(Workflow.createWorkflowForEntity(trustCheck, workflow.getId()));

        Attachment couponAttachment = createSuburbanCouponAttachment(email, order);
        workflowRepository.save(Workflow.createWorkflowForEntity(couponAttachment, workflow.getId()));

        notificationRepository.save(email);
        return workflow.getId();
    }

    private Notification createEmail(Order order) {
        SuburbanOrderItem orderItem = OrderCompatibilityUtils.getSuburbanOrderItem(order);
        SuburbanProviderBase provider = suburbanEnvProvider.createEnv(orderItem).createSuburbanProvider();

        var emailInfo = new EmailChannelInfo();
        emailInfo.setCampaign(provider.getOrderConfirmedCampaign());
        emailInfo.setTarget(order.getEmail());
        emailInfo.setArguments(provider.getConfirmedMailSenderArguments());

        var email = Notification.createEmailNotification(order, emailInfo);
        Duration timeout = provider.getPreparingAttachmentsTimeout();
        email.setPreparingAttachmentsTill(Instant.now().plus(timeout));

        return email;
    }

    private Attachment createFiscalReceiptAttachment(Notification email, Order order) {
        TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
        var receipt = invoice.getFiscalReceipts().stream()
                .filter(i -> i.getReceiptType() == FiscalReceiptType.ACQUIRE)
                .collect(Collectors.toList());
        Preconditions.checkState(receipt.size() == 1, "exactly 1 receipt is expected");
        return Attachment.createFiscalReceiptAttachment(email, "Check_" + email.getOrderPrettyId() + ".pdf",
                NotificationFileTypes.PDF, false, receipt.get(0).getId());
    }

    private Attachment createSuburbanCouponAttachment(Notification email, Order order) {
        SuburbanOrderItem orderItem = OrderCompatibilityUtils.getSuburbanOrderItem(order);
        SuburbanProviderBase provider = suburbanEnvProvider.createEnv(orderItem).createSuburbanProvider();
        return provider.createCouponAttachment(email);
    }
}
