package ru.yandex.travel.orders.services.suburban.environment;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.repository.mock.MockSuburbanOrderItemRepository;
import ru.yandex.travel.orders.services.DeduplicationService;
import ru.yandex.travel.orders.workflows.orderitem.suburban.SuburbanProperties;
import ru.yandex.travel.suburban.partners.aeroexpress.AeroexpressClient;
import ru.yandex.travel.suburban.partners.movista.MovistaClient;
import ru.yandex.travel.train.partners.im.ImClient;

@Service
@Slf4j
public class SuburbanOrderItemEnvProvider {
    /* Initializer of SuburbanOrderItemEnv in a specific context (when orderItem is already known) */

    private final Environment environment;
    private final MovistaClient realMovistaClient;
    private final ImClient realImClient;
    private final AeroexpressClient realAeroexpressClient;
    private final MockSuburbanOrderItemRepository mockSuburbanOrderItemRepository;
    private final PlatformTransactionManager transactionManager;
    private final DeduplicationService deduplicationService;
    private final SuburbanProperties props;

    public SuburbanOrderItemEnvProvider(
            Environment environment,
            MovistaClient realMovistaClient,
            @Qualifier("imSuburbanClient") ImClient realImClient,
            AeroexpressClient realAeroexpressClient,
            MockSuburbanOrderItemRepository mockSuburbanOrderItemRepository,
            PlatformTransactionManager transactionManager,
            DeduplicationService deduplicationService,
            SuburbanProperties props
    ) {
        this.environment = environment;
        this.realMovistaClient = realMovistaClient;
        this.realImClient = realImClient;
        this.realAeroexpressClient = realAeroexpressClient;
        this.mockSuburbanOrderItemRepository = mockSuburbanOrderItemRepository;
        this.transactionManager = transactionManager;
        this.deduplicationService = deduplicationService;
        this.props = props;
    }

    public SuburbanOrderItemEnv createEnv(SuburbanOrderItem orderItem) {
        if (environment.acceptsProfiles("prod")) {  // disable test context in production
            return new SuburbanOrderItemEnv(
                    orderItem, props, deduplicationService, realMovistaClient, realImClient, realAeroexpressClient);
        }

        return new SuburbanOrderItemEnvWithTestContext(
                orderItem,
                props,
                realMovistaClient,
                realImClient,
                realAeroexpressClient,
                createTransactionTemplate(),
                mockSuburbanOrderItemRepository,
                deduplicationService);
    }

    private TransactionTemplate createTransactionTemplate() {
        DefaultTransactionDefinition defaultTransactionDefinition = new DefaultTransactionDefinition();
        defaultTransactionDefinition.setName("SuburbanOrderItemEnvTestContext");
        defaultTransactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        defaultTransactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition transactionDefinition = new DefaultTransactionDefinition(defaultTransactionDefinition);
        return new TransactionTemplate(transactionManager, transactionDefinition);
    }
}
