package ru.yandex.travel.orders.services.suburban.environment;

import java.util.function.Function;

import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.orders.commons.proto.TSuburbanTestContext;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.mock.suburban.MockAeroexpressClient;
import ru.yandex.travel.orders.entities.mock.suburban.MockImSuburbanClient;
import ru.yandex.travel.orders.entities.mock.suburban.MockMovistaClient;
import ru.yandex.travel.orders.entities.mock.suburban.MockSuburbanOrderItem;
import ru.yandex.travel.orders.repository.mock.MockSuburbanOrderItemRepository;
import ru.yandex.travel.orders.services.DeduplicationService;
import ru.yandex.travel.orders.workflows.orderitem.suburban.SuburbanProperties;
import ru.yandex.travel.suburban.partners.aeroexpress.AeroexpressClient;
import ru.yandex.travel.suburban.partners.movista.MovistaClient;
import ru.yandex.travel.train.partners.im.ImClient;

@Slf4j
public class SuburbanOrderItemEnvWithTestContext extends SuburbanOrderItemEnv {
    /*
    A class to choose behavior related to SuburbanOrderItem and its flow.

    Eg: Choose MockMovistaClient instead of DefaultMovistaClient if order item has test context specified
    */

    private final TSuburbanTestContext testContext;
    private final MovistaClient realMovistaClient;
    private final ImClient realImClient;
    private final AeroexpressClient realAeroexpressClient;
    private final TransactionTemplate transactionTemplate;
    private final MockSuburbanOrderItemRepository mockSuburbanOrderItemRepository;

    private MockSuburbanOrderItem mockItem;

    SuburbanOrderItemEnvWithTestContext(
            SuburbanOrderItem orderItem,
            SuburbanProperties props,
            MovistaClient realMovistaClient,
            ImClient realImClient,
            AeroexpressClient realAeroexpressClient,
            TransactionTemplate transactionTemplate,
            MockSuburbanOrderItemRepository mockSuburbanOrderItemRepository,
            DeduplicationService deduplicationService
    ) {
        super(orderItem, props, deduplicationService, realMovistaClient, realImClient, realAeroexpressClient);
        this.testContext = (TSuburbanTestContext) orderItem.getTestContext();
        this.realMovistaClient = realMovistaClient;
        this.realImClient = realImClient;
        this.realAeroexpressClient = realAeroexpressClient;
        this.transactionTemplate = transactionTemplate;
        this.mockSuburbanOrderItemRepository = mockSuburbanOrderItemRepository;

        initMockState();
    }

    private void initMockState() {
        if (testContext == null) {
            return;
        }
        withTx((unused) -> {
            mockItem = mockSuburbanOrderItemRepository.getByOrderItemId(orderItem.getId());
            if (mockItem == null) {
                mockItem = new MockSuburbanOrderItem();
                mockItem.setOrderItemId(orderItem.getId());
                mockSuburbanOrderItemRepository.save(mockItem);
            }
            return null;
        });
    }

    @Override
    public MovistaClient getMovistaClient() {
        if (testContext == null) {
            return realMovistaClient;
        }
        return new MockMovistaClient(
                mockItem, orderItem.getReservation(), testContext,
                transactionTemplate, mockSuburbanOrderItemRepository);
    }

    @Override
    public ImClient getImSuburbanClient() {
        if (testContext == null) {
            return realImClient;
        }
        return new MockImSuburbanClient(
                mockItem, orderItem.getReservation(), testContext,
                transactionTemplate, mockSuburbanOrderItemRepository);
    }

    @Override
    public AeroexpressClient getAeroexpressClient() {
        if (testContext == null) {
            return realAeroexpressClient;
        }
        return new MockAeroexpressClient(
                mockItem, orderItem.getReservation(), testContext,
                transactionTemplate, mockSuburbanOrderItemRepository);
    }

    private <ReqT, RspT> RspT withTx(Function<ReqT, RspT> handler) {
        return transactionTemplate.execute((ignored) -> handler.apply(null));
    }
}
