package ru.yandex.travel.orders.services.suburban.providers;

import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Date;

import com.fasterxml.jackson.databind.node.POJONode;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.notifications.suburban.AeroexpressConfirmedMailSenderArgs;
import ru.yandex.travel.orders.services.payments.model.TrustTerminalForPartner;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnv;
import ru.yandex.travel.orders.workflows.orderitem.suburban.SuburbanProperties;
import ru.yandex.travel.suburban.model.AeroexpressReservation;
import ru.yandex.travel.suburban.partners.aeroexpress.AeroexpressClient;
import ru.yandex.travel.suburban.partners.aeroexpress.model.RequestTicketRequestItem2;
import ru.yandex.travel.suburban.partners.aeroexpress.model.RequestTickets3;
import ru.yandex.travel.suburban.partners.aeroexpress.model.RequestTicketsRequest2;
import ru.yandex.travel.suburban.partners.aeroexpress.model.RequestTicketsResponse;
import ru.yandex.travel.suburban.partners.aeroexpress.model.TicketHistory2;

@Slf4j
public class AeroexpressProvider extends SuburbanProviderBase {

    private final SimpleDateFormat REQUEST_DATE_FORMATTER = new SimpleDateFormat("yyyy-MM-dd'T00:00:00+00:00'");

    public AeroexpressProvider(SuburbanOrderItem orderItem, SuburbanProperties props, SuburbanOrderItemEnv env) {
        super(orderItem, props, env);
    }

    public FiscalItemType getFiscalItemType() {
        return FiscalItemType.SUBURBAN_AEROEXPRESS_TICKET;
    }

    public boolean enableAddBillingTransactions() { return false; }

    private SuburbanProperties.AeroexpressProps getProperties() {
        return props.getProviders().getAeroexpress();
    }

    protected SuburbanProperties.ProviderProps getProviderProps() {
        return getProperties().getCommon();
    }

    public BookResult bookOrder() {
        AeroexpressClient aeroexpressClient = env.getAeroexpressClient();
        RequestTickets3 requestTicketsRequest = createRequestTicketsRequest();
        RequestTicketsResponse requestTicketsResponse = aeroexpressClient.requestTickets(requestTicketsRequest);

        aeroexpressReservation().setOrderId(requestTicketsResponse.getOrderId());
        Money price = Money.of(requestTicketsResponse.getSumm(), ProtoCurrencyUnit.RUB);
        Instant expiresAt = Instant.now().plus(getProperties().getOrderExpireTimeout());

        return BookResult.builder()
                .price(price)
                .expiresAt(expiresAt)
                .build();
    }

    public ConfirmResult confirmOrder() {
        AeroexpressClient aeroexpressClient = env.getAeroexpressClient();
        Long providerOrderId = aeroexpressReservation().getOrderId();

        aeroexpressClient.payOrder(providerOrderId);
        TicketHistory2 ticketResponse = aeroexpressClient.getOrderTickets(providerOrderId);

        aeroexpressReservation().setTicketId(ticketResponse.getTicketId());
        aeroexpressReservation().setTicketUrl(ticketResponse.getTicketUrl());
        aeroexpressReservation().setTariff(ticketResponse.getTariff());
        aeroexpressReservation().setStDepart(ticketResponse.getStDepart());
        aeroexpressReservation().setTripDate(ticketResponse.getTripDate());
        aeroexpressReservation().setValidUntil(ticketResponse.getValidUntil());

        return ConfirmResult.builder()
                .ticketNumber(ticketResponse.getTicketId().toString())
                .build();
    }

    public void onConfirm() {
        if (orderItem.getTestContext() != null) {
            throw new UnsupportedOperationException("Test context for aeroexpress is not implemented");
        }
    }

    private RequestTickets3 createRequestTicketsRequest() {
        var item = new ArrayList<RequestTicketRequestItem2>();
        item.add(new RequestTicketRequestItem2());

        var items = new RequestTicketsRequest2.Items();
        items.setItem(item);

        var request2 = new RequestTicketsRequest2();
        request2.setMenuId(aeroexpressReservation().getMenuId());
        request2.setOrderType(aeroexpressReservation().getOrderType());
        Date date = Date.from(aeroexpressReservation().getDate().atStartOfDay(ZoneId.of("Etc/UTC")).toInstant());
        request2.setDepartDate(REQUEST_DATE_FORMATTER.format(date));
        request2.setItems(items);

        var request = new RequestTickets3();
        request.setRequest(request2);

        return request;
    }

    public TrustTerminalForPartner getTerminalForPartner() {
        // АЭ заведен на отдельном сервисе, там нет выбора терминала - он один
        return null;
    }

    @Override
    protected String getCouponAttachmentName() {
        return "E-ticket.pdf";
    }

    public byte[] getCouponAttachmentData() {
        AeroexpressClient aeroexpressClient = env.getAeroexpressClient();
        String ticketPdfUrl = aeroexpressReservation().getTicketUrl() + "&type=pdf";
        return aeroexpressClient.downloadTicket(ticketPdfUrl);
    }

    public POJONode getConfirmedMailSenderArguments() {
        AeroexpressConfirmedMailSenderArgs mailArgs = AeroexpressConfirmedMailSenderArgs.builder()
                .mainName("")
                .route(aeroexpressReservation().getStDepart())
                .orderPrice(orderItem.getPayload().getPrice().getNumberStripped())
                .ticketCount(1)
                .receiptFile(String.format("Check_%s.pdf", orderItem.getOrder().getPrettyId()))
                .build();

        return new POJONode(mailArgs);
    }

    private AeroexpressReservation aeroexpressReservation() {
        return orderItem.getPayload().getAeroexpressReservation();
    }
}
