package ru.yandex.travel.orders.services.suburban.providers;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.UUID;

import com.fasterxml.jackson.databind.node.POJONode;
import com.google.common.base.Strings;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.services.DeduplicationService;
import ru.yandex.travel.orders.services.payments.model.TrustTerminalForPartner;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnv;
import ru.yandex.travel.orders.workflows.orderitem.suburban.SuburbanProperties;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.suburban.exceptions.SuburbanBadStatusException;
import ru.yandex.travel.suburban.exceptions.SuburbanRetryableException;
import ru.yandex.travel.suburban.model.ImReservation;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.train.model.CarType;
import ru.yandex.travel.train.model.DocumentType;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.train.model.Sex;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.ImClientParseException;
import ru.yandex.travel.train.partners.im.ImClientRetryableException;
import ru.yandex.travel.train.partners.im.model.OrderFullCustomerRequest;
import ru.yandex.travel.train.partners.im.model.OrderReservationBlankRequest;
import ru.yandex.travel.train.partners.im.model.OrderReservationTicketBarcodeRequest;
import ru.yandex.travel.train.partners.im.model.OrderReservationTicketBarcodeResponse;
import ru.yandex.travel.train.partners.im.model.RailwayPassengerRequest;
import ru.yandex.travel.train.partners.im.model.RailwayReservationRequest;
import ru.yandex.travel.train.partners.im.model.ReservationCreateRequest;
import ru.yandex.travel.train.partners.im.model.ReservationCreateResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemBlank;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;
import ru.yandex.travel.workflow.exceptions.RetryableException;

public class ImProvider extends SuburbanProviderBase {
    private static final Integer IM_ORDER_ITEM_ID = 0;  // always 0 because we have only one item in suburban orders

    public ImProvider(SuburbanOrderItem orderItem, SuburbanProperties props,
                      SuburbanOrderItemEnv env, DeduplicationService deduplicationService) {
        super(orderItem, props, env);
        this.deduplicationService = deduplicationService;
    }

    private DeduplicationService deduplicationService;

    public FiscalItemType getFiscalItemType() {
        return FiscalItemType.SUBURBAN_IM_TICKET;
    }

    private SuburbanProperties.ImProps getProperties() {
        return props.getProviders().getIm();
    }

    protected SuburbanProperties.ProviderProps getProviderProps() {
        return getProperties().getCommon();
    }

    public BookResult bookOrder() {
        ImClient imClient = env.getImSuburbanClient();
        ReservationCreateRequest reservationReq = createBookRequest();
        ReservationCreateResponse reservationResponse;
        try {
            reservationResponse = imClient.reservationCreate(reservationReq, null);
        }
        catch (ImClientRetryableException | ImClientParseException ex) {
            throw new SuburbanRetryableException(ex.getMessage(), ex);
        }
        imReservation().setOrderId(reservationResponse.getOrderId());
        Instant validUntil = ImHelpers.fromLocalDateTime(reservationResponse.getConfirmTill(), ImHelpers.MSK_TZ);
        return BookResult.builder()
                .price(Money.of(reservationResponse.getAmount(), ProtoCurrencyUnit.RUB))
                .expiresAt(validUntil).build();
    }

    public ConfirmResult confirmOrder() {
        Integer orderId = imReservation().getOrderId();
        ImClient imClient = env.getImSuburbanClient();
        SuburbanReservation.WorkflowData workflowData = orderItem.getPayload().getWorkflowData();

        try {
            // confirm можно вызывать только один раз
            if (tryRegisterAtMostOnce(workflowData.getConfirmRequestUuid()))
                imClient.reservationConfirm(orderId);

            // get order item
            OrderInfoResponse infoResponse = imClient.orderInfo(orderId);
            OrderItemResponse imOrderItem = infoResponse.getOrderItems().get(0);
            OrderItemBlank blankData = imOrderItem.getOrderItemBlanks().get(0);

            if (imOrderItem.getSimpleOperationStatus() == ImOperationStatus.IN_PROCESS) {
                throw new SuburbanBadStatusException("IM reservation order item simpleOperationStatus is InProcess");
            }

            imReservation().setOrderItemId(imOrderItem.getOrderItemId());
            imReservation().setOrderItemBlankId(blankData.getOrderItemBlankId());
            imReservation().setTicketNumber(blankData.getBlankNumber());
        }
        catch (ImClientException ex) {
            throw new SuburbanRetryableException(ex.getMessage(), ex);
        }
        return ConfirmResult.builder().ticketNumber(imReservation().getTicketNumber()).build();
    }

    public Integer getGetTicketBarcodeMaxAttempts() {
        return getProps().getProviders().getIm().getGetTicketBarcodeMaxAttempts();
    }

    public Duration getGetTicketBarcodeRetryingTime() {
        return getProps().getProviders().getIm().getGetTicketBarcodeRetryingTime();
    }

    public void getTicketBarcode() {
        ImClient imClient = env.getImSuburbanClient();
        try {
            // get ticket number & barcode
            OrderReservationTicketBarcodeRequest barcodeReq = new OrderReservationTicketBarcodeRequest(
                    imReservation().getOrderItemBlankId(), imReservation().getOrderItemId());
            OrderReservationTicketBarcodeResponse barcodeResponse = imClient.orderReservationTicketBarcode(
                    barcodeReq, getProperties().getGetTicketBarcodeTimeout());

            if (Strings.isNullOrEmpty(barcodeResponse.getTicketBarcodeText()))
                throw new ImClientParseException("Empty ticketBody in IM confirm response");
            imReservation().setTicketBody(barcodeResponse.getTicketBarcodeText());
        }
        catch (ImClientException ex) {
            throw new SuburbanRetryableException(ex.getMessage(), ex);
        }
    }

    private boolean tryRegisterAtMostOnce(UUID operationUuid) {
        try {
            deduplicationService.registerAtMostOnceCall(operationUuid);
            return true;
        } catch (IllegalStateException e) {
            return false;
        }
    }

    private ReservationCreateRequest createBookRequest() {
        // значимые для бронирования данные
        var reserveItem = new RailwayReservationRequest();
        reserveItem.setCarType(CarType.SEDENTARY);
        reserveItem.setDepartureDate(imReservation().getDate().atStartOfDay());
        reserveItem.setOriginCode(imReservation().getStationFromExpressId().toString());
        reserveItem.setDestinationCode(imReservation().getStationToExpressId().toString());
        reserveItem.setTrainNumber(imReservation().getTrainNumber());
        String imProvider = imReservation().getImProvider();
        reserveItem.setProvider(imProvider == null ? "P6" : imProvider);

        // фейковые данные покупателя - минимально необходимые для запроса в ИМ
        var customer = new OrderFullCustomerRequest();
        customer.setDocumentType(DocumentType.RUSSIAN_PASSPORT);
        customer.setDocumentNumber("0000000000");
        customer.setFirstName("НеУказано");
        customer.setLastName("НеУказано");
        customer.setMiddleName("НеУказано");
        customer.setSex(Sex.MALE);
        customer.setCitizenshipCode("RU");

        // фейковые данные пассажира - минимально необходимые для запроса в ИМ
        var requestPassenger = new RailwayPassengerRequest();
        requestPassenger.setCategory(PassengerCategory.ADULT);
        requestPassenger.setContactEmailOrPhone(getProperties().getFakePassengerEmail());

        // собираем запрос
        ReservationCreateRequest request = new ReservationCreateRequest();
        request.setCustomers(new ArrayList<>());
        request.getCustomers().add(customer);
        customer.setIndex(0);

        reserveItem.setPassengers(new ArrayList<>());
        reserveItem.getPassengers().add(requestPassenger);
        requestPassenger.setOrderCustomerIndex(0);

        request.setReservationItems(new ArrayList<>());
        request.getReservationItems().add(reserveItem);
        reserveItem.setIndex(0);

        return request;
    }

    public TrustTerminalForPartner getTerminalForPartner() {
        return TrustTerminalForPartner.IM;
    }

    @Override
    public byte[] getCouponAttachmentData() {
        ImClient imClient = env.getImSuburbanClient();
        var blankRequest = new OrderReservationBlankRequest(imReservation().getOrderId(), IM_ORDER_ITEM_ID);
        try {
            return imClient.orderReservationBlank(blankRequest, getProperties().getDownloadBlankTimeout());
        } catch (ImClientRetryableException ex) {
            throw new RetryableException(ex);
        }
    }

    public POJONode getConfirmedMailSenderArguments() {
        return new POJONode(getConfirmedMailBaseArgsBuilder().build());
    }

    private ImReservation imReservation() {
        return orderItem.getPayload().getImReservation();
    }
}
