package ru.yandex.travel.orders.services.suburban.providers;

import java.time.ZoneId;
import java.time.ZonedDateTime;

import com.fasterxml.jackson.databind.node.POJONode;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.commons.proto.TSuburbanTestContext;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.notifications.suburban.SuburbanConfirmedMailSenderArgs;
import ru.yandex.travel.orders.services.payments.model.TrustTerminalForPartner;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnv;
import ru.yandex.travel.orders.workflows.orderitem.suburban.SuburbanProperties;
import ru.yandex.travel.suburban.exceptions.SuburbanRetryableException;
import ru.yandex.travel.suburban.model.MovistaReservation;
import ru.yandex.travel.suburban.partners.movista.MovistaClient;
import ru.yandex.travel.suburban.partners.movista.MovistaErrorCode;
import ru.yandex.travel.suburban.partners.movista.MovistaModel;
import ru.yandex.travel.suburban.partners.movista.MovistaOrderStatus;
import ru.yandex.travel.suburban.partners.movista.exceptions.MovistaParseResponseException;
import ru.yandex.travel.suburban.partners.movista.exceptions.MovistaRequestException;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class MovistaProvider extends SuburbanProviderBase {
    public MovistaProvider(SuburbanOrderItem orderItem, SuburbanProperties props, SuburbanOrderItemEnv env) {
        super(orderItem, props, env);
    }

    public FiscalItemType getFiscalItemType() {
        return FiscalItemType.SUBURBAN_MOVISTA_TICKET;
    }

    protected SuburbanProperties.ProviderProps getProviderProps() {
        return props.getProviders().getMovista().getCommon();
    }

    public BookResult bookOrder() {
        MovistaClient movistaClient = env.getMovistaClient();
        MovistaModel.BookRequest bookReq = createBookRequest();
        MovistaModel.OrderResponse orderResponse = movistaClient.book(bookReq);

        movistaReservation().setOrderId(orderResponse.getOrderId());

        ZonedDateTime validUntil = orderResponse.getValidDate().atZone(ZoneId.of("Europe/Moscow"));
        return BookResult.builder()
                .price(Money.of(orderResponse.getPrice(), ProtoCurrencyUnit.RUB))
                .expiresAt(validUntil.toInstant()).build();
    }

    public ConfirmResult confirmOrder() {
        MovistaClient movistaClient = env.getMovistaClient();
        MovistaModel.OrderResponse orderResponse;
        try {
            MovistaModel.ConfirmRequest confirmReq = createConfirmRequest();
            orderResponse = movistaClient.confirm(confirmReq);
        } catch (MovistaRequestException ex) {
            if (ex.errorCode == MovistaErrorCode.ORDER_STATUS_INVALID) {
                // https://st.yandex-team.ru/RASPFRONT-9563
                // check if order is already confirmed
                orderResponse = movistaClient.orderInfo(orderItem.getReservation().getMovistaReservation().getOrderId());
                if (orderResponse.getStatus() != MovistaOrderStatus.CONFIRMED) {
                    throw ex;
                }
            } else {
                throw ex;
            }
        }

        Integer ticketNumber = orderResponse.getTicketNumber();
        String ticketBody = orderResponse.getTicketBody();

        if (Strings.isNullOrEmpty(ticketBody) || ticketNumber == 0)
            throw new MovistaParseResponseException("Empty ticketBody or ticketNumber in Movista confirm response");

        movistaReservation().setTicketNumber(ticketNumber);
        movistaReservation().setTicketBody(ticketBody);

        return ConfirmResult.builder()
                .ticketNumber(movistaReservation().getTicketNumber().toString()).build();
    }

    public void onConfirm() {
        if (orderItem.getTestContext() != null) {
            var testContext = (TSuburbanTestContext) orderItem.getTestContext();
            if (!Strings.isNullOrEmpty(testContext.getTicketBody())) {
                movistaReservation().setTicketBody(testContext.getTicketBody());
            }
        }
    }

    private MovistaModel.BookRequest createBookRequest() {
        return MovistaModel.BookRequest.builder()
                .date(movistaReservation().getDate())
                .fromExpressId(movistaReservation().getStationFromExpressId())
                .toExpressId(movistaReservation().getStationToExpressId())
                .fareId(movistaReservation().getFareId())
                .email(orderItem.getOrder().getEmail())
                .phone(orderItem.getOrder().getPhone()).build();
    }

    private MovistaModel.ConfirmRequest createConfirmRequest() {
        return MovistaModel.ConfirmRequest.builder()
                .orderId(movistaReservation().getOrderId())
                .txId(orderItem.getId().toString()).build();
    }

    public TrustTerminalForPartner getTerminalForPartner() {
        return TrustTerminalForPartner.MOVISTA;
    }

    public byte[] getCouponAttachmentData() {
        MovistaClient movistaClient = env.getMovistaClient();
        try {
            return movistaClient.getBlankPdf(orderItem.getPayload().getMovistaReservation().getOrderId());
        } catch (SuburbanRetryableException ex) {
            throw new RetryableException(ex);
        }
    }

    public POJONode getConfirmedMailSenderArguments() {
        SuburbanConfirmedMailSenderArgs.SuburbanConfirmedMailSenderArgsBuilder
                mailArgsBuilder = getConfirmedMailBaseArgsBuilder();
        mailArgsBuilder.validatorOnStation(orderItem.getPayload().getMovistaReservation().hasValidator());

        return new POJONode(mailArgsBuilder.build());
    }

    private MovistaReservation movistaReservation() {
        return orderItem.getPayload().getMovistaReservation();
    }
}
