package ru.yandex.travel.orders.services.suburban.providers;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.time.Duration;
import java.time.Instant;

import com.fasterxml.jackson.databind.node.POJONode;
import lombok.Builder;
import lombok.Data;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.entities.notifications.Notification;
import ru.yandex.travel.orders.entities.notifications.suburban.SuburbanConfirmedMailSenderArgs;
import ru.yandex.travel.orders.entities.notifications.suburban.SuburbanCouponAttachmentProviderData;
import ru.yandex.travel.orders.services.NotificationFileTypes;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.payments.model.TrustTerminalForPartner;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnv;
import ru.yandex.travel.orders.workflow.orderitem.suburban.proto.TBeforeReservation;
import ru.yandex.travel.orders.workflow.orderitem.suburban.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.orderitem.suburban.proto.TReservationCommit;
import ru.yandex.travel.orders.workflows.orderitem.suburban.SuburbanProperties;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.workflow.logging.TEventSolomonProperties;

@Data
public class SuburbanProviderBase {
    /*
    Реализация логики поставщика услуг по продаже билетов на электрички.

    Например: Мовиста и ИМ - имеют общий флоу, но со своей спецификой,
    которая реализуется в подклассах SuburbanProviderBase.
    */
    protected final SuburbanOrderItem orderItem;
    protected final SuburbanProperties props;
    protected final SuburbanOrderItemEnv env;

    public FiscalItemType getFiscalItemType() {
        throw new UnsupportedOperationException();
    }

    protected SuburbanProperties.ProviderProps getProviderProps() {
        throw new UnsupportedOperationException();
    }

    public BookResult bookOrder() {
        throw new UnsupportedOperationException();
    }

    public ConfirmResult confirmOrder() {
        throw new UnsupportedOperationException();
    }

    public void getTicketBarcode() {}

    public void onConfirm() {}

    public TBeforeReservation getTBeforeReservationMessage() {
        return TBeforeReservation.newBuilder().setSolomonProperties(getEventSolomonProperties()).build();
    }

    public TReservationCommit getTReservationCommitMessage() {
        return TReservationCommit.newBuilder().setSolomonProperties(getEventSolomonProperties()).build();
    }

    public TConfirmationCommit getTConfirmationCommitMessage() {
        return TConfirmationCommit.newBuilder().setSolomonProperties(getEventSolomonProperties()).build();
    }

    private TEventSolomonProperties getEventSolomonProperties() {
        SuburbanReservation reservation = orderItem.getPayload();
        String tag = MessageFormat.format(
                "{0}.{1}", reservation.getProvider().getValue(), reservation.getCarrier().getValue());
        return TEventSolomonProperties.newBuilder().setTag(tag).build();
    }

    public TrustTerminalForPartner getTerminalForPartner() {
        throw new UnsupportedOperationException();
    }

    public Long getBillingClientId() {
        return getProviderProps().getBillingClientId();
    }

    public Integer getBookMaxAttempts() {
        return getProviderProps().getBookMaxAttempts();
    }

    public Integer getConfirmMaxAttempts() { return getProviderProps().getConfirmMaxAttempts();}

    public Duration getConfirmRetryingTime() { return getProviderProps().getConfirmRetryingTime();}

    public Integer getGetTicketBarcodeMaxAttempts() { return 0; }

    public Duration getGetTicketBarcodeRetryingTime() { return null; }

    public String getFiscalTitle() {
        return getProviderProps().getFiscalInfo().getTitle();
    }

    public String getFiscalInn() {
        return getProviderProps().getFiscalInfo().getInn();
    }

    public Duration getPaymentClearingDelay() {return getProviderProps().getPaymentClearingDelay(); }

    // Записывать транзакции биллинга в таблицу YT
    public boolean enableAddBillingTransactions() { return true; }

    public Money getRewardFee(Money price) {
        return price.multiply(calcRate(getProviderProps().getFinancialEvents().getRewardFee()));
    }

    public Money getRewardFeeRefund(Money price) {
        return price.multiply(calcRate(getProviderProps().getFinancialEvents().getRewardFeeRefund()));
    }

    private BigDecimal calcRate(BigDecimal fee) {
        // fee * (1 + vat)
        return fee.multiply(getProviderProps().getFinancialEvents().getVat().add(BigDecimal.valueOf(1)));
    }

    public String getOrderConfirmedCampaign() {
        return getProviderProps().getMail().getOrderConfirmedCampaign();
    }

    public Duration getPreparingAttachmentsTimeout() {
        return getProviderProps().getMail().getPreparingAttachmentsTimeout();
    }

    protected String getCouponAttachmentName() {
        return "Контрольный купон.pdf";
    }

    public Attachment createCouponAttachment(Notification email) {
        return Attachment.createSuburbanCouponAttachmentData(
                email, getCouponAttachmentName(), NotificationFileTypes.PDF, false,
                SuburbanCouponAttachmentProviderData.builder().orderItemId(orderItem.getId()).build());
    }

    public byte[] getCouponAttachmentData() {
        throw new UnsupportedOperationException();
    }

    protected SuburbanConfirmedMailSenderArgs.SuburbanConfirmedMailSenderArgsBuilder getConfirmedMailBaseArgsBuilder() {
        SuburbanReservation reservation = orderItem.getPayload();
        return SuburbanConfirmedMailSenderArgs.builder()
                .ticketNumber(reservation.getProviderTicketNumber())
                .orderPrice(reservation.getPrice().getNumberStripped())
                .travelDate(NotificationHelper.humanDate(reservation.getDepartureDateTime().toLocalDateTime()))
                .stationFromTitle(reservation.getStationFrom().getTitleDefault())
                .stationToTitle(reservation.getStationTo().getTitleDefault());
    }

    public POJONode getConfirmedMailSenderArguments() {
        throw new UnsupportedOperationException();
    }

    @Data
    @Builder
    public static class BookResult {
        Money price;
        Instant expiresAt;
    }

    @Data
    @Builder
    public static class ConfirmResult {
        String ticketNumber;
    }
}
