package ru.yandex.travel.orders.services.support

import com.google.common.base.Preconditions
import com.google.common.base.Strings
import com.google.common.collect.ImmutableMap
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.stereotype.Service
import org.springframework.transaction.annotation.Propagation
import org.springframework.transaction.annotation.Transactional
import ru.yandex.travel.hotels.common.orders.ConfirmationInfo
import ru.yandex.travel.hotels.common.orders.HotelItinerary
import ru.yandex.travel.orders.commons.proto.EServiceType
import ru.yandex.travel.orders.entities.HotelOrder
import ru.yandex.travel.orders.entities.HotelOrderItem
import ru.yandex.travel.orders.entities.Order
import ru.yandex.travel.orders.entities.OrderItem
import ru.yandex.travel.orders.entities.support.SuccessfulHotelOrderNotification
import ru.yandex.travel.orders.entities.support.UnsuccessfulHotelOrderNotification
import ru.yandex.travel.orders.repository.HotelOrderRepository
import ru.yandex.travel.orders.repository.support.SuccessfulHotelOrderNotificationRepository
import ru.yandex.travel.orders.repository.support.UnsuccessfulHotelOrderNotificationRepository
import ru.yandex.travel.orders.services.support.SupportEmailService.BookingAttempt
import java.time.Instant
import java.time.LocalDate
import java.util.*
import java.util.stream.Collectors

@Service
@EnableConfigurationProperties(HotelsSupportProperties::class)
open class HotelsSupportNotificationService @Autowired constructor(
    private val hotelOrderRepository: HotelOrderRepository,
    private val successfulHotelOrderNotificationRepository: SuccessfulHotelOrderNotificationRepository,
    private val unsuccessfulHotelOrderNotificationRepository: UnsuccessfulHotelOrderNotificationRepository,
    private val emailService: SupportEmailService,
    private val hotelsSupportProperties: HotelsSupportProperties,
) {

    val log: Logger = LoggerFactory.getLogger(javaClass)

    @Transactional
    open fun scheduleNewSuccessfulOrders() {
        val since = Instant.now().minus(hotelsSupportProperties.successfulOrders.maxLookBehind)
        log.debug("Running new successful orders scan; since={}", since)
        val copied = scheduleNewConfirmedOrders(
            since,
            LocalDate.now().plusDays(hotelsSupportProperties.successfulOrders.maxCheckinDateForward),
            hotelsSupportProperties.successfulOrders.maxOrdersForHotel
        )
        if (copied.isNotEmpty()) {
            log.info("New successful orders have been scheduled for support notification: {} object(s)", copied.size)
        }
    }

    @Transactional
    open fun scheduleNewConfirmedOrders(since: Instant, maxCheckinDate: LocalDate, maxOrdersForHotel: Long): List<HotelOrderItem> {
        val orderItems = successfulHotelOrderNotificationRepository.selectNewConfirmedOrders(since)
        val permalinks: List<Long> = orderItems.map { it.hotelItinerary.orderDetails.permalink }
        val ordersCountByPermalink = successfulHotelOrderNotificationRepository.countForPermalinks(permalinks)
            .associateBy ({it.permalink}, {(it.count).toLong()}).toMutableMap()

        val insertedItems = mutableListOf<HotelOrderItem>()
        for (orderItem in orderItems) {
            val permalink = orderItem.hotelItinerary.orderDetails.permalink
            val ordersCount = ordersCountByPermalink.getOrDefault(permalink, 0)
            if (ordersCount >= maxOrdersForHotel) {
                continue
            }

            val checkinDate = orderItem.hotelItinerary.orderDetails.checkinDate;
            if (checkinDate >= maxCheckinDate) {
                continue
            }

            val orderNotification = SuccessfulHotelOrderNotification()
            orderNotification.order = orderItem.order as HotelOrder
            orderNotification.permalink = orderItem.hotelItinerary.orderDetails.permalink

            successfulHotelOrderNotificationRepository.save(orderNotification);
            ordersCountByPermalink[permalink] = ordersCount + 1
            insertedItems.add(orderItem)
        }
        return insertedItems
    }

    @Transactional
    open fun findPendingSuccessfulHotelOrderNotifications(): Set<UUID> {
        log.debug("Running pending successful order notifications scan")
        return successfulHotelOrderNotificationRepository.findBySentAtIsNull().stream()
                .map { obj: SuccessfulHotelOrderNotification -> obj.orderId }
                .collect(Collectors.toSet())
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW)
    open fun sendSuccessfulHotelOrderNotification(orderId: UUID) {
        log.info("Preparing successful order notification for {}", orderId)
        val notification = successfulHotelOrderNotificationRepository.getOne(orderId)
        Preconditions.checkArgument(notification.sentAt == null,
                "The notification has already been sent at %s",
                notification.sentAt)
        val order = hotelOrderRepository.getOne(orderId)
        val itinerary = getItinerary(order)
        Preconditions.checkArgument(itinerary.guests.size > 0,
                "At least 1 guest is expected: %s", itinerary.guests)
        val mainGuest = itinerary.guests[0]
        val userName = mainGuest.fullName
        val phone = itinerary.customerPhone
        val email = itinerary.customerEmail
        val hotel = itinerary.orderDetails.hotelName
        val bookingId = getBookingId(itinerary.confirmation)
        val partner = getPartnerName(order)
        emailService.sendSuccessfulOrderNotification(order.prettyId,
                userName, phone, email, hotel, bookingId, partner, order.createdAt,
                itinerary.orderDetails.checkinDate,
                itinerary.orderDetails.checkoutDate, itinerary.orderDetails.hotelPhone,
                itinerary.orderDetails.ratePlanDetails)
        notification.sentAt = Instant.now()
        successfulHotelOrderNotificationRepository.save(notification)
    }

    @Transactional
    open fun scheduleNewUnsuccessfulOrders() {
        val since = Instant.now().minus(hotelsSupportProperties.unsuccessfulOrders.maxLookBehind)
        val till = Instant.now().minus(hotelsSupportProperties.unsuccessfulOrders.notificationDelay)
        log.debug("Running new unsuccessful orders scan; since={}, till={}", since, till)
        val copied = unsuccessfulHotelOrderNotificationRepository.copyNewUnconfirmedOrders(since, till)
        if (copied > 0) {
            log.info("New unsuccessful orders have been scheduled for support notification: {} object(s)", copied)
        }
    }

    @Transactional
    open fun findPendingUnsuccessfulHotelOrderNotifications(): Collection<List<UUID>> {
        log.debug("Running pending unsuccessful order notifications scan")
        val groups: MutableMap<SameAuthGroup, MutableList<UUID>> = HashMap()
        for (notification in unsuccessfulHotelOrderNotificationRepository.findNotSentOrdered()) {
            val group = SameAuthGroup(notification.passportId, notification.sessionKey)
            groups.computeIfAbsent(group) { key: SameAuthGroup? -> ArrayList() }.add(notification.orderId)
        }
        if (!groups.isEmpty()) {
            log.info("Pending unsuccessful order group notifications have been found: {}", groups.size)
        }
        return groups.values
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW)
    open fun sendUnsuccessfulHotelOrderNotification(orderIds: List<UUID>) {
        log.info("Preparing unsuccessful order group notification for orders {}", orderIds)
        val notifications: MutableList<UnsuccessfulHotelOrderNotification> = ArrayList()
        val attempts: MutableList<BookingAttempt> = ArrayList()
        for (orderId in orderIds) {
            val notification = unsuccessfulHotelOrderNotificationRepository.getOne(orderId)
            Preconditions.checkArgument(notification.sentAt == null,
                    "The notification has already been sent at %s",
                    notification.sentAt)
            notifications.add(notification)
            val order = hotelOrderRepository.getOne(orderId)
            val itinerary = getItinerary(order)
            Preconditions.checkArgument(itinerary.guests.size > 0,
                    "At least 1 guest is expected: %s", itinerary.guests)
            val mainGuest = itinerary.guests[0]
            val details = itinerary.orderDetails

            val bookingAttempt = BookingAttempt()
            bookingAttempt.prettyId = order.prettyId
            bookingAttempt.userName = mainGuest.fullName
            bookingAttempt.phone = itinerary.customerPhone
            bookingAttempt.email = itinerary.customerEmail
            bookingAttempt.hotel = if (details != null) details.hotelName else "<no order details>"
            bookingAttempt.partner = getPartnerName(order)
            bookingAttempt.createdAt = order.createdAt
            bookingAttempt.state = order.state
            attempts.add(bookingAttempt)
        }

        // TRAVELORDERSUP-1178: temporary disabling emails, the notifications shouldn't be re-sent later;
        // don't forget to re-enabled the sendUnsuccessfulOrderGroupNotification test
        //emailService.sendUnsuccessfulOrderNotification(attempts);
        for (notification in notifications) {
            notification.sentAt = Instant.now()
            unsuccessfulHotelOrderNotificationRepository.save(notification)
        }
    }

    private fun getItinerary(order: HotelOrder): HotelItinerary {
        val orderItem = order.orderItems[0]
        Preconditions.checkArgument(orderItem is HotelOrderItem,
                "Unexpected order item type: ", orderItem.publicType)
        return (orderItem as HotelOrderItem).hotelItinerary
    }

    data class SameAuthGroup(
        val passportId: String,
        val sessionKey: String
    )

    companion object {
        val PARTNER_NAMES: Map<EServiceType, String> = ImmutableMap.builder<EServiceType, String>()
                .put(EServiceType.PT_EXPEDIA_HOTEL, "Expedia")
                .put(EServiceType.PT_DOLPHIN_HOTEL, "Дельфин")
                .put(EServiceType.PT_TRAVELLINE_HOTEL, "TravelLine")
                .put(EServiceType.PT_BNOVO_HOTEL, "BNovo")
                .put(EServiceType.PT_BRONEVIK_HOTEL, "Броневик")
                .build()

        @JvmStatic
        fun getBookingId(confirmation: ConfirmationInfo?): String? {
            var hotelId: String? = null
            var partnerId: String? = null
            if (confirmation != null) {
                hotelId = confirmation.hotelConfirmationId
                partnerId = confirmation.partnerConfirmationId
            }
            if (!Strings.isNullOrEmpty(hotelId)) {
                return hotelId
            }
            return if (!Strings.isNullOrEmpty(partnerId)) {
                partnerId
            } else "<no booking id>"
        }

        @JvmStatic
        fun getPartnerName(order: Order): String {
            // at the moment we expect exactly 1 partner here but let's not fail in other cases
            return java.lang.String.join(", ", order.orderItems.stream()
                    .map { oi: OrderItem -> PARTNER_NAMES[oi.publicType] ?: oi.publicType.name }
                    .collect(Collectors.toSet()))
        }
    }
}
