package ru.yandex.travel.orders.services.support;

import java.time.Duration;

import javax.annotation.PostConstruct;
import javax.validation.Valid;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;

import com.google.common.base.Preconditions;
import lombok.Data;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.validation.annotation.Validated;

@Data
@ConfigurationProperties(value = "hotels-support", ignoreUnknownFields = false)
@Validated
public class HotelsSupportProperties {
    @NotEmpty
    private String senderEmail;
    @NotNull
    @Valid
    private SuccessfulOrders successfulOrders;
    @NotNull
    @Valid
    private UnsuccessfulOrders unsuccessfulOrders;
    @NotNull
    @Valid
    private PotentiallyEligibleMirOrders potentiallyEligibleMirOrders;

    @PostConstruct
    void validate() {
        unsuccessfulOrders.validate();
    }

    @Data
    @Validated
    public static class SuccessfulOrders {
        @NotNull
        private String email;
        @NotNull
        private Duration maxLookBehind;
        @NotNull
        private Long maxCheckinDateForward = 30L;
        @NotNull
        private Long maxOrdersForHotel = 5L;
        @NotNull
        private Boolean notificationsEnabled;

        public String getEmail() {
            return email;
        }

        public void setEmail(String email) {
            this.email = email;
        }

        public Duration getMaxLookBehind() {
            return maxLookBehind;
        }

        public void setMaxLookBehind(Duration maxLookBehind) {
            this.maxLookBehind = maxLookBehind;
        }

        public Long getMaxCheckinDateForward() {
            return maxCheckinDateForward;
        }

        public void setMaxCheckinDateForward(Long maxCheckinDateForward) {
            this.maxCheckinDateForward = maxCheckinDateForward;
        }

        public Long getMaxOrdersForHotel() {
            return maxOrdersForHotel;
        }

        public void setMaxOrdersForHotel(Long maxOrdersForHotel) {
            this.maxOrdersForHotel = maxOrdersForHotel;
        }

        public Boolean getNotificationsEnabled() {
            return notificationsEnabled;
        }

        public void setNotificationsEnabled(Boolean notificationsEnabled) {
            this.notificationsEnabled = notificationsEnabled;
        }
    }

    @Data
    @Validated
    public static class UnsuccessfulOrders {
        @NotNull
        private String email;
        @NotNull
        private Duration maxLookBehind;
        @NotNull
        private Duration notificationDelay;
        @NotNull
        private Boolean notificationsEnabled;

        void validate() {
            Preconditions.checkArgument(maxLookBehind.toMillis() > notificationDelay.toMillis(),
                    "Illegal unsuccessfulOrders configuration: maxLookBehind should be greater than notificationDelay; " +
                            "maxLookBehind=%s, notificationDelay=%s", maxLookBehind, notificationDelay);
        }

        public String getEmail() {
            return email;
        }

        public void setEmail(String email) {
            this.email = email;
        }

        public Duration getMaxLookBehind() {
            return maxLookBehind;
        }

        public void setMaxLookBehind(Duration maxLookBehind) {
            this.maxLookBehind = maxLookBehind;
        }

        public Duration getNotificationDelay() {
            return notificationDelay;
        }

        public void setNotificationDelay(Duration notificationDelay) {
            this.notificationDelay = notificationDelay;
        }

        public Boolean getNotificationsEnabled() {
            return notificationsEnabled;
        }

        public void setNotificationsEnabled(Boolean notificationsEnabled) {
            this.notificationsEnabled = notificationsEnabled;
        }
    }

    @Data
    @Validated
    public static class PotentiallyEligibleMirOrders {
        @NotNull
        private String email;

        public String getEmail() {
            return email;
        }

        public void setEmail(String email) {
            this.email = email;
        }
    }

    public String getSenderEmail() {
        return senderEmail;
    }

    public void setSenderEmail(String senderEmail) {
        this.senderEmail = senderEmail;
    }

    public SuccessfulOrders getSuccessfulOrders() {
        return successfulOrders;
    }

    public void setSuccessfulOrders(SuccessfulOrders successfulOrders) {
        this.successfulOrders = successfulOrders;
    }

    public UnsuccessfulOrders getUnsuccessfulOrders() {
        return unsuccessfulOrders;
    }

    public void setUnsuccessfulOrders(UnsuccessfulOrders unsuccessfulOrders) {
        this.unsuccessfulOrders = unsuccessfulOrders;
    }

    public PotentiallyEligibleMirOrders getPotentiallyEligibleMirOrders() {
        return potentiallyEligibleMirOrders;
    }

    public void setPotentiallyEligibleMirOrders(PotentiallyEligibleMirOrders potentiallyEligibleMirOrders) {
        this.potentiallyEligibleMirOrders = potentiallyEligibleMirOrders;
    }
}
