package ru.yandex.travel.orders.services.support;

import java.time.Instant;
import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.repository.HotelOrderRepository;
import ru.yandex.travel.workflow.single_operation.SingleOperationRunner;

@Service
@Slf4j
@RequiredArgsConstructor
public class MirSupportNotificationSender implements SingleOperationRunner<UUID, Void> {
    private final HotelOrderRepository hotelOrderRepository;
    private final SupportEmailService emailService;

    @Override
    public Class<UUID> getInputClass() {
        return UUID.class;
    }

    @Override
    public Void runOperation(UUID orderId) {
        HotelOrder order = hotelOrderRepository.getOne(orderId);
        try (var ignored = NestedMdc.forEntity(order)) {
            log.info("Preparing mir order notification");
            OrderItem orderItem = order.getOrderItems().get(0);
            Preconditions.checkArgument(orderItem instanceof HotelOrderItem,
                    "Unexpected order item type: " + orderItem.getClass().getSimpleName());
            HotelItinerary itinerary = ((HotelOrderItem) orderItem).getHotelItinerary();
            Instant lastRefundableAt = itinerary.getRefundRules().getFullyRefundableTill();
            Preconditions.checkArgument(itinerary.getGuests().size() > 0,
                    "At least 1 guest is expected: %s", itinerary.getGuests());
            Guest mainGuest = itinerary.getGuests().get(0);
            String userName = mainGuest.getFullName();
            String phone = order.getPhone();
            String email = order.getEmail();
            String hotel = itinerary.getOrderDetails().getHotelName();
            String bookingId = HotelsSupportNotificationService.getBookingId(itinerary.getConfirmation());
            String partner = HotelsSupportNotificationService.getPartnerName(order);
            emailService.sendPotentiallyEligibleMirOrderNotification(order.getPrettyId(), orderId.toString(),
                    userName, phone, email, hotel, bookingId, partner, order.getCreatedAt(),
                    itinerary.getOrderDetails().getCheckinDate(),
                    itinerary.getOrderDetails().getCheckoutDate(), lastRefundableAt);
            log.info("Notification sent");
            return null;
        }
    }
}
