package ru.yandex.travel.orders.services.support;

import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.List;

import com.google.common.base.Strings;
import lombok.Builder;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;

import static java.util.stream.Collectors.joining;

@Service
@RequiredArgsConstructor
@Slf4j
public class SupportEmailService {
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm z");
    private static final ZoneId MSK_TIMEZONE = ZoneId.of("Europe/Moscow");

    private final JavaMailSender mailSender;
    private final HotelsSupportProperties hotelsSupportProperties;

    private static String format(Instant instant) {
        if (instant == null) {
            return "null";
        }
        return DATE_TIME_FORMATTER.format(instant.atZone(MSK_TIMEZONE));
    }

    public void sendSuccessfulOrderNotification(String orderId, String userName, String phone, String email,
                                                String hotelName, String bookingId, String partner, Instant createdAt,
                                                LocalDate checkinDate, LocalDate checkoutDate, String hotelPhone,
                                                String ratePlanRequirements) {
        String subject = String.format("Обзвон отелей: новый успешный заказ %s", orderId);
        String message = String.format("Привет! \n" +
                        "Создалась новая бронь:\n" +
                        "- ФИО пользователя: %s\n" +
                        "- Номер брони: %s\n" +
                        "- Телефон пользователя: %s\n" +
                        "- Почта пользователя: %s\n" +
                        "- Название отеля: %s\n" +
                        "- Партнёр: %s\n" +
                        "- Дата создания заказа: %s\n" +
                        "- Дата заезда: %s\n" +
                        "- Дата выезда: %s\n" +
                        "- Телефон отеля: %s\n" +
                        "- Специфические требования от отеля: %s\n\n" +
                        "Нужно позвонить в отель: \n" +
                        "   - проверить, что с бронью всё хорошо, и пользователя ждут\n" +
                        "   - уточнить, нужна ли справка о том, что человек не болеет covid-19, справка какого образца" +
                        " нужна и за сколько дней\n" +
                        "\n" +
                        "Звони через Алтай: https://altay.yandex-team.ru/dialer",
                userName, bookingId, phone, email, hotelName, partner, format(createdAt), checkinDate.toString(),
                checkoutDate.toString(), hotelPhone, ratePlanRequirements);
        sendEmail(
                hotelsSupportProperties.getSenderEmail(),
                hotelsSupportProperties.getSuccessfulOrders().getEmail(),
                subject, message
        );
    }

    public void sendPotentiallyEligibleMirOrderNotification(String orderPrettyId, String orderId, String userName,
                                                            String phone, String email,
                                                            String hotelName, String bookingId, String partner,
                                                            Instant createdAt,
                                                            LocalDate checkinDate, LocalDate checkoutDate, Instant lastRefund) {
        String subject = String.format("Обзвон пользователей: бронь %s подходит под программу лояльности Мир, но " +
                "оплачена не Миром", orderPrettyId);
        String message = String.format("Привет! \n" +
                        "Создалась новая бронь %s, подходящая под программу лояльности Мир:\n" +
                        "- ФИО пользователя: %s\n" +
                        "- Номер брони: %s\n" +
                        "- Заказ в админке: https://travel-orders-admin.yandex-team.ru/%s\n" +
                        "- Телефон пользователя: %s\n" +
                        "- Почта пользователя: %s\n" +
                        "- Название отеля: %s\n" +
                        "- Партнёр: %s\n" +
                        "- Дата создания заказа: %s\n" +
                        "- Дата заезда: %s\n" +
                        "- Дата выезда: %s\n\n" +
                        "Нужно позвонить пользователю и сказать: \n" +
                        "Здравствуйте! Меня зовут ... , я из команды Яндекс.Путешествий. Вы сделали у нас бронь на " +
                        "отель «%s». Хочу вас проинформировать, что данный отель участвует в акции кешбэк" +
                        " 20%% при оплате картой Мир. Вы оплатили другой картой. Если вы хотите получить кешбэк, вам " +
                        "нужно сдать бронь и оформить заново, но оплатить картой Мир. Бесплатная отмена брони по " +
                        "вашему заказу возможна до %s. Чтобы кешбэк пришел, вы также должны быть " +
                        "зарегистрированы в программе лояльности МИР https://privetmir.ru/register/ \n\n" +
                        "Звони через Алтай: https://altay.yandex-team.ru/dialer",
                orderPrettyId, userName, bookingId, orderId, phone, email, hotelName, partner, format(createdAt), checkinDate.toString(),
                checkoutDate.toString(), hotelName, format(lastRefund));
        sendEmail(
                hotelsSupportProperties.getSenderEmail(),
                hotelsSupportProperties.getPotentiallyEligibleMirOrders().getEmail(),
                subject, message
        );
    }

    public void sendUnsuccessfulOrderNotification(List<BookingAttempt> bookingAttempts) {
        String subject = String.format("Позвонить пользователю по %s %s",
                bookingAttempts.size() > 1 ? "неуспешным заказам" : "неуспешному заказу",
                bookingAttempts.stream().map(BookingAttempt::getPrettyId).collect(joining(", "))
        );
        StringBuilder message = new StringBuilder("Привет! \n" +
                "Пользователь пытался забронировать отель, но у него не получилось. Твоя задача узнать причину и " +
                "приложить максимум усилий для того, чтобы бронирование состоялось. \n" +
                "Скрипт разговора: \n" +
                "1. Здороваемся с пользователем: здравствуйте, меня зовут ..., я из компании Яндекс.  Видим, что вы " +
                "недавно оформляли бронирование у нас на сайте, но что-то пошло так. Скажите,  с какими сложностями " +
                "столкнулись?\n" +
                "2. Я готов(а) помочь, ответить на дополнительные вопросы или подсказать  как оформить бронирование " +
                "повторно. \n" +
                "\n" +
                "3. Если клиент согласен на нашу помощь - помогаем пошагово оформить бронь.\n" +
                "Если появились доп вопросы по размещению - предлагаем связаться с отелем и уточнить. Договариваемся " +
                "о времени следующего звонка клиенту.\n" +
                "Все данные подробно фиксируем в комментарии к тикету.\n" +
                "В случае успешной брони, не забудь внести данные в табличку для выплаты премии :) \n" +
                "https://wiki.yandex-team.ru/eva/Yandex.Travel/QAhotel/Tablica-Podbor-otelja/"
        );
        for (BookingAttempt booking : bookingAttempts) {
            message.append(String.format("\n" +
                            "Данные по заказу %s:\n" +
                            "- ФИО пользователя: %s\n" +
                            "- Телефон пользователя: %s\n" +
                            "- Почта пользователя: %s\n" +
                            "- Название отеля: %s\n" +
                            "- Партнёр: %s\n" +
                            "- Дата создания заказа: %s\n" +
                            "- Состояние заказа: %s\n",
                    booking.getPrettyId(), booking.getUserName(), booking.getPhone(), booking.getEmail(),
                    booking.getHotel(), booking.getPartner(), format(booking.getCreatedAt()), booking.getState()
            ));
        }
        message.append("\nЗвони через Алтай: https://altay.yandex-team.ru/dialer");
        sendEmail(
                hotelsSupportProperties.getSenderEmail(),
                hotelsSupportProperties.getUnsuccessfulOrders().getEmail(),
                subject, message.toString()
        );
    }

    private void sendEmail(String from, String to, String subject, String message) {
        if (Strings.isNullOrEmpty(to)) {
            log.warn("No support email is configured, skipping the notification; subject={}", subject);
            return;
        }
        log.info("Sending support email to {}: {}", to, subject);
        SimpleMailMessage smm = new SimpleMailMessage();
        smm.setFrom(from);
        smm.setTo(to.split(","));
        smm.setSubject(subject);
        smm.setText(message);
        mailSender.send(smm);
    }

    @Data
    @Builder
    public static class BookingAttempt {
        public String prettyId;
        public String userName;
        public String phone;
        public String email;
        public String hotel;
        public String partner;
        public Instant createdAt;
        public EHotelOrderState state;
    }
}
