package ru.yandex.travel.orders.services.takeout;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotCarrier;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOriginDestination;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotSegment;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotVariant;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingPriceInfo;
import ru.yandex.avia.booking.partners.gateways.model.booking.ClientInfo;
import ru.yandex.avia.booking.partners.gateways.model.booking.TravellerInfo;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.repository.OrderTakeoutRepository;
import ru.yandex.travel.orders.services.EDisplayOrderStateMapper;
import ru.yandex.travel.takeout.models.AviaTakeoutOrder;
import ru.yandex.travel.takeout.models.AviaTakeoutPassenger;
import ru.yandex.travel.takeout.models.AviaTakeoutSegment;
import ru.yandex.travel.takeout.models.TakeoutOrderStatus;

import static java.util.stream.Collectors.toList;
import static ru.yandex.travel.orders.commons.proto.EDisplayOrderType.DT_AVIA;

@Service
@RequiredArgsConstructor
public class AviaOrderTakeoutService {
    private final OrderTakeoutRepository orderTakeoutRepository;

    public List<AviaTakeoutOrder> getOrdersByUser(String passportId) {
        try (Stream<Order> ordersStream = orderTakeoutRepository.streamOrdersOwnedByUser(passportId, DT_AVIA)) {
            return ordersStream.map(o -> convertOrder((AeroflotOrder) o)).collect(toList());
        }
    }

    private AviaTakeoutOrder convertOrder(AeroflotOrder order) {
        AeroflotOrderItem orderItem = order.getAeroflotOrderItem();
        AeroflotServicePayload payload = orderItem.getPayload();
        BookingPriceInfo costs = payload.getActualCosts();
        AeroflotOrderRef bookingRef = payload.getBookingRef();
        ClientInfo clientInfo = payload.getClientInfo();
        return AviaTakeoutOrder.builder()
                .prettyId(order.getPrettyId())
                .createdAt(order.getCreatedAt())
                .status(TakeoutOrderStatus.fromDisplayState(EDisplayOrderStateMapper.fromOrder(order)))
                .price(costs != null ? costs.getTotalAmount() : null)
                .segments(convertSegments(payload.getVariant()))
                .passengers(convertPassengers(payload.getTravellers()))
                .pnr(bookingRef != null ? bookingRef.getPnr() : null)
                .tickets(payload.getTicketsFlatList())
                .email(clientInfo.getEmail())
                .phone(clientInfo.getPhone())
                .build();
    }

    public static List<List<AviaTakeoutSegment>> convertSegments(AeroflotVariant variant) {
        if (variant == null || variant.getOriginDestinations() == null || variant.getSegments() == null) {
            return null;
        }
        Map<String, List<AviaTakeoutSegment>> odSegments = new HashMap<>();
        for (AeroflotSegment segment : variant.getSegments()) {
            odSegments.computeIfAbsent(segment.getOriginDestinationId(), id -> new ArrayList<>())
                    .add(convertSegments(segment));
        }
        List<List<AviaTakeoutSegment>> orderedSegments = new ArrayList<>();
        for (AeroflotOriginDestination originDestination : variant.getOriginDestinations()) {
            orderedSegments.add(odSegments.get(originDestination.getId()));
        }
        return orderedSegments;
    }

    private static AviaTakeoutSegment convertSegments(AeroflotSegment segment) {
        AeroflotCarrier mCarrier = segment.getMarketingCarrier();
        AeroflotCarrier oCarrier = segment.getOperatingCarrier();
        return AviaTakeoutSegment.builder()
                .marketingCarrier(mCarrier != null ? mCarrier.getAirlineId() : null)
                .marketingFlightNumber(mCarrier != null ? mCarrier.getFlightNumber() : null)
                .operatingCarrier(oCarrier != null ? oCarrier.getAirlineId() : null)
                .operatingFlightNumber(oCarrier != null ? oCarrier.getFlightNumber() : null)
                .departureAirport(segment.getDeparture().getAirportCode())
                .departureAt(segment.getDeparture().getDateTime())
                .arrivalAirport(segment.getArrival().getAirportCode())
                .arrivalAt(segment.getArrival().getDateTime())
                .build();
    }

    public static List<AviaTakeoutPassenger> convertPassengers(List<TravellerInfo> travellers) {
        return travellers.stream()
                .map(t -> AviaTakeoutPassenger.builder()
                        .category(t.getCategory().getValue())
                        .firstName(t.getFirstName())
                        .middleName(t.getMiddleName())
                        .lastName(t.getLastName())
                        .dateOfBirth(t.getDateOfBirth())
                        .nationalityCode(t.getNationalityCode())
                        .sex(t.getSex().getValue())
                        .documentNumber(t.getDocumentNumber())
                        .documentType(t.getDocumentType().getValue())
                        .documentValidTill(t.getDocumentValidTill())
                        .build())
                .collect(toList());
    }
}
