package ru.yandex.travel.orders.services.takeout;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.model.booking.BookingPriceInfo;
import ru.yandex.avia.booking.partners.gateways.model.booking.ClientInfo;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.orders.RefundInfo;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.repository.OrderTakeoutRepository;
import ru.yandex.travel.orders.services.EDisplayOrderStateMapper;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.takeout.models.AviaTakeoutOrder;
import ru.yandex.travel.takeout.models.AviaTakeoutOrderItem;
import ru.yandex.travel.takeout.models.HotelTakeoutOrder;
import ru.yandex.travel.takeout.models.HotelTakeoutOrderItem;
import ru.yandex.travel.takeout.models.SuburbanTakeoutOrderItem;
import ru.yandex.travel.takeout.models.TakeoutOrder;
import ru.yandex.travel.takeout.models.TakeoutOrderStatus;
import ru.yandex.travel.takeout.models.TrainOrder;
import ru.yandex.travel.takeout.models.TrainTakeoutOrderItem;
import ru.yandex.travel.train.model.TrainReservation;

@Service
@RequiredArgsConstructor
public class GenericOrderTakeoutService {
    private static final Map<EDisplayOrderType, String> DISPLAY_ORDER_TYPE_MAP = Map.of(
            EDisplayOrderType.DT_TRAIN, TrainOrder.TRAIN_ORDER_TYPE,
            EDisplayOrderType.DT_AVIA, AviaTakeoutOrder.AVIA_ORDER_TYPE,
            EDisplayOrderType.DT_HOTEL, HotelTakeoutOrder.HOTEL_ORDER_TYPE
    );
    private final OrderTakeoutRepository orderTakeoutRepository;

    public List<TakeoutOrder> getOrdersByUser(String passportId) {
        try (Stream<Order> ordersStream = orderTakeoutRepository.streamOrdersOwnedByUser(passportId)) {
            return ordersStream.map(this::getFromOrder).collect(Collectors.toList());
        }
    }

    private TakeoutOrder getFromOrder(Order order) {
        var res = new TakeoutOrder();
        res.setTrainItems(new ArrayList<>());
        res.setHotelItems(new ArrayList<>());
        res.setAviaItems(new ArrayList<>());
        res.setSuburbanItems(new ArrayList<>());
        res.setCreatedAt(order.getCreatedAt());
        String takeoutOrderType = DISPLAY_ORDER_TYPE_MAP.get(order.getDisplayType());
        res.setType(takeoutOrderType);
        res.setStatus(TakeoutOrderStatus.fromDisplayState(EDisplayOrderStateMapper.fromOrder(order)));
        res.setPrettyId(order.getPrettyId());
        res.setPayAmount(Money.zero(order.getCurrency()));
        var invoice = order.getCurrentInvoice();
        if (invoice != null) {
            BigDecimal originalSum = invoice.getInvoiceItems().stream()
                    .map(x -> x.getOriginalPrice() == null ? x.getPrice() : x.getOriginalPrice())
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
            res.setPayAmount(Money.of(originalSum, order.getCurrency()));
            BigDecimal actualSum = invoice.calculateCurrentAmount();
            res.setRefundAmount(Money.of(originalSum.subtract(actualSum), order.getCurrency()));
        }
        for (var item : order.getOrderItems()) {
            switch (item.getPublicType()) {
                case PT_TRAIN:
                    res.getTrainItems().add(mapTrainOrderItem(order, (TrainOrderItem) item));
                    break;
                case PT_BNOVO_HOTEL:
                case PT_EXPEDIA_HOTEL:
                case PT_DOLPHIN_HOTEL:
                case PT_BRONEVIK_HOTEL:
                case PT_TRAVELLINE_HOTEL:
                    res.getHotelItems().add(mapHotelOrderItem((HotelOrderItem) item));
                    break;
                case PT_FLIGHT:
                    res.getAviaItems().add(mapAeroflotOrderItem((AeroflotOrderItem) item));
                    break;
                case PT_SUBURBAN:
                    res.getSuburbanItems().add(mapSuburbanOrderItem((SuburbanOrderItem) item));
                    break;
            }
        }
        return res;
    }

    private AviaTakeoutOrderItem mapAeroflotOrderItem(AeroflotOrderItem orderItem) {
        AeroflotServicePayload payload = orderItem.getPayload();
        BookingPriceInfo costs = payload.getActualCosts();
        AeroflotOrderRef bookingRef = payload.getBookingRef();
        ClientInfo clientInfo = payload.getClientInfo();
        return AviaTakeoutOrderItem.builder()
                .price(costs != null ? costs.getTotalAmount() : null)
                .segments(AviaOrderTakeoutService.convertSegments(payload.getVariant()))
                .passengers(AviaOrderTakeoutService.convertPassengers(payload.getTravellers()))
                .pnr(bookingRef != null ? bookingRef.getPnr() : null)
                .tickets(payload.getTicketsFlatList())
                .email(clientInfo.getEmail())
                .phone(clientInfo.getPhone())
                .build();
    }

    private TrainTakeoutOrderItem mapTrainOrderItem(Order order, TrainOrderItem item) {
        var res = new TrainTakeoutOrderItem();
        TrainReservation payload = item.getPayload();
        res.setTrainNumber(payload.getTrainNumber());
        res.setCarNumber(payload.getCarNumber());
        res.setCarType(payload.getCarType());
        res.setDepartureStation(payload.getUiData().getStationFromTitle());
        res.setArrivalStation(payload.getUiData().getStationToTitle());
        res.setDepartureTime(payload.getDepartureTime());
        res.setArrivalTime(payload.getArrivalTime());
        res.setReservationNumber(payload.getReservationNumber());
        res.setPassengers(payload.getPassengers().stream()
                .map(p -> TrainOrderTakeoutService.getFromTrainPassenger(payload, p))
                .collect(Collectors.toList()));
        res.setPayAmount(item.totalCostAfterReservation());
        res.setFeeAmount(payload.getPassengers().stream()
                .filter(x -> x.getTicket() != null)
                .map(x -> x.getTicket().getFeeAmount())
                .reduce(Money::add).orElse(Money.zero(order.getCurrency())));
        res.setRefundAmount(item.getTrainTicketRefunds().stream()
                .filter(x -> x.getState() == ETrainTicketRefundState.RS_REFUNDED)
                .map(x -> x.getPayload().calculateActualRefundSum())
                .reduce(Money::add).orElse(Money.zero(order.getCurrency())));
        return res;
    }

    private HotelTakeoutOrderItem mapHotelOrderItem(HotelOrderItem orderItem) {
        HotelItinerary itinerary = orderItem.getHotelItinerary();
        OrderDetails details = itinerary.getOrderDetails();
        RefundInfo refund = itinerary.getRefundInfo();
        return HotelTakeoutOrderItem.builder()
                .price(itinerary.getFiscalPrice())
                .refundedAt(refund != null ?
                        HotelOrderTakeoutService.convertUtcLocalDateToInstant(refund.getRefundDateTime()) : null)
                .refundedSum(refund != null ? refund.getRefund().asMoney() : null)
                .hotelName(details.getHotelName())
                .hotelRoomName(details.getRoomName())
                .checkIn(details.getCheckinDate())
                .checkOut(details.getCheckoutDate())
                .guests(HotelOrderTakeoutService.convertGuests(itinerary.getGuests()))
                .phone(itinerary.getCustomerPhone())
                .email(itinerary.getCustomerEmail())
                .build();
    }

    private SuburbanTakeoutOrderItem mapSuburbanOrderItem(SuburbanOrderItem orderItem) {
        SuburbanReservation reservation = orderItem.getPayload();
        return SuburbanTakeoutOrderItem.builder()
                .date(reservation.getDepartureDateTime().toLocalDate())
                .stationFrom(reservation.getStationFrom().getTitleDefault())
                .stationTo(reservation.getStationTo().getTitleDefault())
                .price(reservation.getPrice())
                .ticketNumber(reservation.getProviderTicketNumber())
                .build();
    }
}
