package ru.yandex.travel.orders.services.takeout;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.orders.RefundInfo;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.repository.OrderTakeoutRepository;
import ru.yandex.travel.orders.services.EDisplayOrderStateMapper;
import ru.yandex.travel.takeout.models.HotelTakeoutGuest;
import ru.yandex.travel.takeout.models.HotelTakeoutOrder;
import ru.yandex.travel.takeout.models.TakeoutOrderStatus;

import static java.util.stream.Collectors.toList;
import static ru.yandex.travel.orders.commons.proto.EDisplayOrderType.DT_HOTEL;

@Service
@RequiredArgsConstructor
public class HotelOrderTakeoutService {
    private final OrderTakeoutRepository orderTakeoutRepository;

    public List<HotelTakeoutOrder> getOrdersByUser(String passportId) {
        try (Stream<Order> ordersStream = orderTakeoutRepository.streamOrdersOwnedByUser(passportId, DT_HOTEL)) {
            return ordersStream.flatMap(o -> convertOrder((HotelOrder) o).stream()).collect(toList());
        }
    }

    private List<HotelTakeoutOrder> convertOrder(HotelOrder order) {
        List<HotelTakeoutOrder> result = new ArrayList<>();
        for (OrderItem orderItem : order.getOrderItems()) {
            HotelOrderItem hotelItem = (HotelOrderItem) orderItem;
            HotelItinerary itinerary = hotelItem.getHotelItinerary();
            OrderDetails details = itinerary.getOrderDetails();
            RefundInfo refund = itinerary.getRefundInfo();
            result.add(HotelTakeoutOrder.builder()
                    .prettyId(order.getPrettyId())
                    .createdAt(order.getCreatedAt())
                    .status(TakeoutOrderStatus.fromDisplayState(EDisplayOrderStateMapper.fromOrder(order)))
                    .price(itinerary.getFiscalPrice())
                    .refundedAt(refund != null ? convertUtcLocalDateToInstant(refund.getRefundDateTime()) : null)
                    .refundedSum(refund != null ? refund.getRefund().asMoney() : null)
                    .hotelName(details.getHotelName())
                    .hotelRoomName(details.getRoomName())
                    .checkIn(details.getCheckinDate())
                    .checkOut(details.getCheckoutDate())
                    .guests(convertGuests(itinerary.getGuests()))
                    .phone(itinerary.getCustomerPhone())
                    .email(itinerary.getCustomerEmail())
                    .build());
        }
        return result;
    }

    public static List<HotelTakeoutGuest> convertGuests(List<Guest> guests) {
        return guests.stream()
                .map(g -> HotelTakeoutGuest.builder()
                        .firstName(g.getFirstName())
                        .lastName(g.getLastName())
                        .child(g.isChild())
                        .age(g.getAge())
                        .build())
                .collect(toList());
    }

    public static Instant convertUtcLocalDateToInstant(LocalDateTime utcDateTime) {
        return utcDateTime.toInstant(ZoneOffset.UTC);
    }
}
