package ru.yandex.travel.orders.services.takeout;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.MoneyRefundState;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.repository.OrderTakeoutRepository;
import ru.yandex.travel.orders.services.EDisplayOrderStateMapper;
import ru.yandex.travel.takeout.models.TakeoutOrderStatus;
import ru.yandex.travel.takeout.models.TrainOrder;
import ru.yandex.travel.takeout.models.TrainOrderPassenger;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;

import static ru.yandex.travel.orders.commons.proto.EDisplayOrderType.DT_TRAIN;

@Service
@RequiredArgsConstructor
public class TrainOrderTakeoutService {
    private final OrderTakeoutRepository orderTakeoutRepository;

    public List<TrainOrder> getOrdersByUser(String passportId) {
        try (Stream<Order> ordersStream = orderTakeoutRepository.streamOrdersOwnedByUser(passportId, DT_TRAIN)) {
            return ordersStream.map(this::getFromOrder).collect(Collectors.toList());
        }
    }

    private TrainOrder getFromOrder(Order order) {
        var res = new TrainOrder();
        TrainOrderItem trainOrderItem = (TrainOrderItem) order.getOrderItems().get(0);
        TrainReservation payload = trainOrderItem.getPayload();

        res.setStatus(TakeoutOrderStatus.fromDisplayState(EDisplayOrderStateMapper.fromOrder(order)));
        res.setTrainNumber(payload.getTrainNumber());
        res.setCarNumber(payload.getCarNumber());
        res.setCarType(payload.getCarType());
        res.setDepartureStation(payload.getUiData().getStationFromTitle());
        res.setArrivalStation(payload.getUiData().getStationToTitle());
        res.setDepartureTime(payload.getDepartureTime());
        res.setArrivalTime(payload.getArrivalTime());

        res.setReservationNumber(payload.getReservationNumber());
        res.setPassengers(payload.getPassengers().stream()
                .map(p -> getFromTrainPassenger(payload, p))
                .collect(Collectors.toList()));
        res.setPayAmount(trainOrderItem.totalCostAfterReservation());
        res.setFeeAmount(payload.getPassengers().stream()
                .filter(x -> x.getTicket() != null)
                .map(x -> x.getTicket().getFeeAmount())
                .reduce(Money::add).orElse(Money.zero(order.getCurrency())));
        res.setRefundAmount(order.getMoneyRefunds().stream()
                .filter(x -> x.getState() == MoneyRefundState.REFUNDED)
                .flatMap(x -> x.getTargetFiscalItems().values().stream())
                .reduce(Money::add).orElse(Money.zero(order.getCurrency())));
        return res;
    }

    public static TrainOrderPassenger getFromTrainPassenger(TrainReservation payload, TrainPassenger source) {
        var res = new TrainOrderPassenger();
        res.setFirstName(source.getFirstName());
        res.setPatronymic(source.getPatronymic());
        res.setLastName(source.getLastName());
        res.setTariffType(source.getTariffType());
        res.setTariffCode(source.getTariffCodeWithFallback());
        res.setSex(source.getSex());
        if (source.getTicket() != null) {
            res.setImBlankStatus(source.getTicket().getImBlankStatus());
            res.setPayAmount(source.getTicket().calculateTotalCost());
            res.setFeeAmount(source.getTicket().getFeeAmount());
        }
        if (payload.getInsuranceStatus() == InsuranceStatus.CHECKED_OUT && source.getInsurance() != null) {
            res.setPayAmount(res.getPayAmount().add(source.getInsurance().getAmount()));
            res.setInsuranceAmount(source.getInsurance().getAmount());
        }
        return res;
    }
}
